import type { Metadata } from "next"
import { Suspense } from "react"
import { EarningsClient } from "@/components/admin/earnings/earnings-client"
import { earningsDataStore } from "@/lib/data/earnings"
import { Skeleton } from "@/components/ui/skeleton"

export const metadata: Metadata = {
  title: "Agent Earnings | Teertham Admin",
  description:
    "View and manage all agent earnings on Teertham platform. Track total earnings, monthly earnings, wallet balance, and withdrawal history for travel agents.",
  openGraph: {
    title: "Agent Earnings | Teertham Admin",
    description: "Comprehensive agent earnings dashboard with detailed financial reports and commission tracking.",
    type: "website",
    siteName: "Teertham",
  },
  twitter: {
    card: "summary",
    title: "Agent Earnings | Teertham Admin",
    description: "Comprehensive agent earnings dashboard with detailed financial reports and commission tracking.",
  },
  robots: {
    index: false,
    follow: false,
  },
}

function EarningsTableSkeleton() {
  return (
    <div className="space-y-6">
      {/* Summary cards skeleton */}
      <div className="grid gap-4 sm:grid-cols-2 lg:grid-cols-4">
        {Array.from({ length: 4 }).map((_, i) => (
          <Skeleton key={i} className="h-24 w-full" />
        ))}
      </div>
      {/* Search skeleton */}
      <div className="flex items-center justify-between">
        <Skeleton className="h-10 w-72" />
        <Skeleton className="h-10 w-24" />
      </div>
      {/* Table skeleton */}
      <Skeleton className="h-96 w-full" />
    </div>
  )
}

export default function AgentEarningPage() {
  const agentEarnings = earningsDataStore.getAgentEarnings()
  const summary = earningsDataStore.getTotalEarningsByType("agent")

  const jsonLd = {
    "@context": "https://schema.org",
    "@type": "WebPage",
    name: "Agent Earnings Dashboard",
    description: "Administrative dashboard for managing agent earnings and commissions on Teertham platform.",
    isPartOf: {
      "@type": "WebSite",
      name: "Teertham Admin",
    },
  }

  return (
    <>
      <script type="application/ld+json" dangerouslySetInnerHTML={{ __html: JSON.stringify(jsonLd) }} />
      <main className="space-y-6">
        <header>
          <h1 className="text-3xl font-bold tracking-tight">Agent Earnings</h1>
          <p className="text-muted-foreground mt-2">
            View earnings reports and commission details for all registered agents
          </p>
        </header>

        <Suspense fallback={<EarningsTableSkeleton />}>
          <EarningsClient data={agentEarnings} summary={summary} userType="agent" />
        </Suspense>
      </main>
    </>
  )
}
