"use client"

import type React from "react"

import { useEffect, useState } from "react"
import { useRouter } from "next/navigation"
import { AgentHeader } from "@/components/agent/agent-header"
import { getClientSession, validateSessionData } from "@/lib/auth/client-session"
import { Card, CardContent, CardHeader, CardTitle, CardDescription } from "@/components/ui/card"
import { Button } from "@/components/ui/button"
import { Input } from "@/components/ui/input"
import { Label } from "@/components/ui/label"
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from "@/components/ui/select"
import { Badge } from "@/components/ui/badge"
import {
  Compass,
  MapPin,
  Calendar,
  Users,
  Car,
  Hotel,
  CheckCircle2,
  ChevronRight,
  Star,
  Clock,
  IndianRupee,
  Sun,
  Sunrise,
  Sunset,
  UtensilsCrossed,
  Camera,
  User,
  Phone,
  Mail,
  Plus,
  Trash2,
  Check,
} from "lucide-react"
import { cn } from "@/lib/utils"

// Mock data for cities with per-head pricing (no category selection for spiritual)
const cities = [
  { id: "varanasi", name: "Varanasi", state: "Uttar Pradesh", pricePerHead: 2500 },
  { id: "ayodhya", name: "Ayodhya", state: "Uttar Pradesh", pricePerHead: 2000 },
  { id: "prayagraj", name: "Prayagraj", state: "Uttar Pradesh", pricePerHead: 1800 },
  { id: "mathura", name: "Mathura", state: "Uttar Pradesh", pricePerHead: 2200 },
  { id: "haridwar", name: "Haridwar", state: "Uttarakhand", pricePerHead: 2800 },
  { id: "rishikesh", name: "Rishikesh", state: "Uttarakhand", pricePerHead: 3000 },
]

// Day itinerary for display purposes only
interface DayItinerary {
  day: number
  title: string
  image: string
  destinations: string[]
  activities: {
    time: string
    icon: React.ReactNode
    title: string
  }[]
}

const cityItineraries: Record<string, DayItinerary[]> = {
  varanasi: [
    {
      day: 1,
      title: "Arrival & Ganga Darshan",
      image: "/placeholder.svg?height=120&width=200",
      destinations: ["Dashashwamedh Ghat", "Kashi Vishwanath Temple"],
      activities: [
        { time: "02:00 PM", icon: <Sun className="h-4 w-4" />, title: "Arrival & Check-in" },
        { time: "05:00 PM", icon: <Star className="h-4 w-4" />, title: "Kashi Vishwanath Darshan" },
        { time: "07:00 PM", icon: <Sunset className="h-4 w-4" />, title: "Ganga Aarti" },
      ],
    },
    {
      day: 2,
      title: "Temple Circuit & Boat Ride",
      image: "/placeholder.svg?height=120&width=200",
      destinations: ["Sankat Mochan", "Durga Temple", "Tulsi Manas Temple"],
      activities: [
        { time: "05:00 AM", icon: <Sunrise className="h-4 w-4" />, title: "Boat Ride on Ganga" },
        { time: "09:00 AM", icon: <Star className="h-4 w-4" />, title: "Temple Visits" },
        { time: "06:00 PM", icon: <Sunset className="h-4 w-4" />, title: "Evening Aarti" },
      ],
    },
    {
      day: 3,
      title: "Sarnath Excursion",
      image: "/placeholder.svg?height=120&width=200",
      destinations: ["Dhamek Stupa", "Sarnath Museum", "Ashoka Pillar"],
      activities: [
        { time: "08:00 AM", icon: <Car className="h-4 w-4" />, title: "Drive to Sarnath" },
        { time: "10:00 AM", icon: <Camera className="h-4 w-4" />, title: "Stupa & Museum Visit" },
        { time: "05:00 PM", icon: <Sunset className="h-4 w-4" />, title: "Return & Ganga Aarti" },
      ],
    },
    {
      day: 4,
      title: "Departure Day",
      image: "/placeholder.svg?height=120&width=200",
      destinations: ["Final Darshan", "Local Shopping"],
      activities: [
        { time: "05:00 AM", icon: <Sunrise className="h-4 w-4" />, title: "Subah-e-Banaras" },
        { time: "09:00 AM", icon: <UtensilsCrossed className="h-4 w-4" />, title: "Breakfast" },
        { time: "12:00 PM", icon: <Car className="h-4 w-4" />, title: "Departure" },
      ],
    },
  ],
  ayodhya: [
    {
      day: 1,
      title: "Ram Janmabhoomi Darshan",
      image: "/placeholder.svg?height=120&width=200",
      destinations: ["Ram Janmabhoomi", "Hanuman Garhi"],
      activities: [
        { time: "12:00 PM", icon: <Sun className="h-4 w-4" />, title: "Arrival" },
        { time: "04:00 PM", icon: <Star className="h-4 w-4" />, title: "Ram Mandir Darshan" },
        { time: "07:00 PM", icon: <Sunset className="h-4 w-4" />, title: "Saryu Aarti" },
      ],
    },
    {
      day: 2,
      title: "Temple Circuit",
      image: "/placeholder.svg?height=120&width=200",
      destinations: ["Kanak Bhawan", "Nageshwarnath Temple", "Treta Ke Thakur"],
      activities: [
        { time: "06:00 AM", icon: <Sunrise className="h-4 w-4" />, title: "Morning Aarti" },
        { time: "09:00 AM", icon: <Star className="h-4 w-4" />, title: "Temple Visits" },
        { time: "05:00 PM", icon: <Camera className="h-4 w-4" />, title: "Saryu Ghat Walk" },
      ],
    },
    {
      day: 3,
      title: "Heritage & Departure",
      image: "/placeholder.svg?height=120&width=200",
      destinations: ["Ram Ki Paidi", "Local Markets"],
      activities: [
        { time: "05:00 AM", icon: <Sunrise className="h-4 w-4" />, title: "Final Darshan" },
        { time: "10:00 AM", icon: <Sun className="h-4 w-4" />, title: "Shopping" },
        { time: "12:00 PM", icon: <Car className="h-4 w-4" />, title: "Departure" },
      ],
    },
    {
      day: 4,
      title: "Extended Exploration",
      image: "/placeholder.svg?height=120&width=200",
      destinations: ["Nearby Temples", "Cultural Sites"],
      activities: [
        { time: "07:00 AM", icon: <Sunrise className="h-4 w-4" />, title: "Morning Rituals" },
        { time: "10:00 AM", icon: <Compass className="h-4 w-4" />, title: "Exploration" },
        { time: "02:00 PM", icon: <Car className="h-4 w-4" />, title: "Departure" },
      ],
    },
  ],
}

// Default itinerary for cities without specific plans
const defaultItinerary: DayItinerary[] = [
  {
    day: 1,
    title: "Arrival & Temple Visit",
    image: "/placeholder.svg?height=120&width=200",
    destinations: ["Main Temple", "City Center"],
    activities: [
      { time: "12:00 PM", icon: <Sun className="h-4 w-4" />, title: "Arrival & Check-in" },
      { time: "04:00 PM", icon: <Star className="h-4 w-4" />, title: "Temple Darshan" },
      { time: "07:00 PM", icon: <Sunset className="h-4 w-4" />, title: "Evening Aarti" },
    ],
  },
  {
    day: 2,
    title: "Temple Circuit",
    image: "/placeholder.svg?height=120&width=200",
    destinations: ["Sacred Sites", "Heritage Locations"],
    activities: [
      { time: "06:00 AM", icon: <Sunrise className="h-4 w-4" />, title: "Morning Darshan" },
      { time: "10:00 AM", icon: <Camera className="h-4 w-4" />, title: "Heritage Walk" },
      { time: "05:00 PM", icon: <Star className="h-4 w-4" />, title: "Evening Prayers" },
    ],
  },
  {
    day: 3,
    title: "Exploration Day",
    image: "/placeholder.svg?height=120&width=200",
    destinations: ["Local Attractions", "Cultural Sites"],
    activities: [
      { time: "08:00 AM", icon: <Sun className="h-4 w-4" />, title: "Breakfast & Start" },
      { time: "10:00 AM", icon: <Compass className="h-4 w-4" />, title: "Sightseeing" },
      { time: "04:00 PM", icon: <UtensilsCrossed className="h-4 w-4" />, title: "Leisure Time" },
    ],
  },
  {
    day: 4,
    title: "Departure Day",
    image: "/placeholder.svg?height=120&width=200",
    destinations: ["Final Darshan", "Departure"],
    activities: [
      { time: "06:00 AM", icon: <Sunrise className="h-4 w-4" />, title: "Morning Prayer" },
      { time: "09:00 AM", icon: <UtensilsCrossed className="h-4 w-4" />, title: "Breakfast" },
      { time: "12:00 PM", icon: <Car className="h-4 w-4" />, title: "Departure" },
    ],
  },
]

// Vehicles data
const vehicles = [
  {
    id: "sedan",
    name: "Sedan",
    image: "/placeholder.svg?height=100&width=160",
    seats: 4,
    pricePerDay: 2500,
    features: ["AC", "Music System"],
  },
  {
    id: "suv",
    name: "SUV",
    image: "/placeholder.svg?height=100&width=160",
    seats: 6,
    pricePerDay: 3500,
    features: ["AC", "Spacious", "Music"],
  },
  {
    id: "innova",
    name: "Innova Crysta",
    image: "/placeholder.svg?height=100&width=160",
    seats: 7,
    pricePerDay: 4500,
    features: ["AC", "Premium", "Captain Seats"],
  },
  {
    id: "tempo",
    name: "Tempo Traveller",
    image: "/placeholder.svg?height=100&width=160",
    seats: 12,
    pricePerDay: 6000,
    features: ["AC", "Pushback Seats", "LCD"],
  },
  {
    id: "bus",
    name: "Mini Bus",
    image: "/placeholder.svg?height=100&width=160",
    seats: 20,
    pricePerDay: 10000,
    features: ["AC", "Mic System", "Spacious"],
  },
]

// Pre-defined hotels with fixed pricing (no star category selection)
interface HotelOption {
  id: string
  name: string
  image: string
  rating: number
  pricePerRoom: number
  amenities: string[]
  location: string
}

const cityHotels: Record<string, HotelOption[]> = {
  varanasi: [
    {
      id: "hotel1",
      name: "Taj Ganges",
      image: "/placeholder.svg?height=120&width=200",
      rating: 5,
      pricePerRoom: 8500,
      amenities: ["Pool", "Spa", "Restaurant", "WiFi"],
      location: "Nadesar Palace Grounds",
    },
    {
      id: "hotel2",
      name: "Radisson Varanasi",
      image: "/placeholder.svg?height=120&width=200",
      rating: 4,
      pricePerRoom: 5500,
      amenities: ["Restaurant", "Gym", "WiFi", "Parking"],
      location: "The Mall, Cantonment",
    },
    {
      id: "hotel3",
      name: "Hotel Surya",
      image: "/placeholder.svg?height=120&width=200",
      rating: 3,
      pricePerRoom: 3200,
      amenities: ["Restaurant", "WiFi", "AC Rooms"],
      location: "Near Dashashwamedh Ghat",
    },
    {
      id: "hotel4",
      name: "Ganga View Guest House",
      image: "/placeholder.svg?height=120&width=200",
      rating: 2,
      pricePerRoom: 1500,
      amenities: ["Ganga View", "WiFi", "Breakfast"],
      location: "Assi Ghat",
    },
  ],
  ayodhya: [
    {
      id: "hotel1",
      name: "Ramada Ayodhya",
      image: "/placeholder.svg?height=120&width=200",
      rating: 4,
      pricePerRoom: 4500,
      amenities: ["Restaurant", "Gym", "WiFi", "Parking"],
      location: "Near Ram Mandir",
    },
    {
      id: "hotel2",
      name: "Hotel Saket",
      image: "/placeholder.svg?height=120&width=200",
      rating: 3,
      pricePerRoom: 2800,
      amenities: ["Restaurant", "WiFi", "AC Rooms"],
      location: "Civil Lines",
    },
    {
      id: "hotel3",
      name: "Shri Ram Bhawan",
      image: "/placeholder.svg?height=120&width=200",
      rating: 2,
      pricePerRoom: 1200,
      amenities: ["Simple Rooms", "Temple Nearby", "Meals"],
      location: "Ram Ki Paidi",
    },
  ],
}

// Default hotels for cities without specific hotels
const defaultHotels: HotelOption[] = [
  {
    id: "default1",
    name: "Premium Hotel",
    image: "/placeholder.svg?height=120&width=200",
    rating: 4,
    pricePerRoom: 4000,
    amenities: ["Restaurant", "WiFi", "AC", "Parking"],
    location: "City Center",
  },
  {
    id: "default2",
    name: "Standard Hotel",
    image: "/placeholder.svg?height=120&width=200",
    rating: 3,
    pricePerRoom: 2500,
    amenities: ["Restaurant", "WiFi", "AC"],
    location: "Near Temple",
  },
  {
    id: "default3",
    name: "Budget Stay",
    image: "/placeholder.svg?height=120&width=200",
    rating: 2,
    pricePerRoom: 1200,
    amenities: ["Basic Amenities", "WiFi"],
    location: "Main Road",
  },
]

// Customer interface
interface Customer {
  id: string
  name: string
  age: string
  gender: string
  phone: string
  email: string
}

// Steps for the booking wizard (no category step)
const steps = [
  { id: 1, title: "Destination", icon: <MapPin className="h-4 w-4" /> },
  { id: 2, title: "Trip Duration", icon: <Calendar className="h-4 w-4" /> },
  { id: 3, title: "Vehicle", icon: <Car className="h-4 w-4" /> },
  { id: 4, title: "Hotel", icon: <Hotel className="h-4 w-4" /> },
  { id: 5, title: "Customer Details", icon: <Users className="h-4 w-4" /> },
  { id: 6, title: "Review & Confirm", icon: <CheckCircle2 className="h-4 w-4" /> },
]

export default function SpiritualBookingPage() {
  const router = useRouter()
  const [agentEmail, setAgentEmail] = useState<string>("agent@teertham.com")
  const [agentName, setAgentName] = useState<string>("Agent")
  const [isLoading, setIsLoading] = useState(true)

  // Wizard state
  const [currentStep, setCurrentStep] = useState(1)
  const [selectedCity, setSelectedCity] = useState("")
  const [tripDays, setTripDays] = useState(2)
  const [passengerCount, setPassengerCount] = useState(2)
  const [selectedVehicle, setSelectedVehicle] = useState("")
  const [selectedHotel, setSelectedHotel] = useState("")
  const [roomCount, setRoomCount] = useState(1)
  const [customers, setCustomers] = useState<Customer[]>([
    { id: "1", name: "", age: "", gender: "", phone: "", email: "" },
  ])

  useEffect(() => {
    const session = getClientSession("agent_session")
    if (!session || !validateSessionData(session) || session.role !== "agent") {
      router.push("/login")
      return
    }
    setAgentEmail(session.email || "agent@teertham.com")
    setAgentName(session.name || "Agent")
    setIsLoading(false)
  }, [router])

  // Get itinerary for selected city
  const getItinerary = () => {
    return cityItineraries[selectedCity] || defaultItinerary
  }

  // Get hotels for selected city
  const getHotels = () => {
    return cityHotels[selectedCity] || defaultHotels
  }

  // Get filtered vehicles based on passenger count
  const filteredVehicles = vehicles.filter((v) => v.seats >= passengerCount)

  // Calculate pricing
  const calculatePricing = () => {
    const city = cities.find((c) => c.id === selectedCity)
    const vehicle = vehicles.find((v) => v.id === selectedVehicle)
    const hotel = getHotels().find((h) => h.id === selectedHotel)

    const tripCostPerHead = city ? city.pricePerHead * tripDays : 0
    const totalTripCost = tripCostPerHead * customers.length
    const vehicleCost = vehicle ? vehicle.pricePerDay * tripDays : 0
    const hotelNights = tripDays > 1 ? tripDays - 1 : 0
    const hotelCost = hotel ? hotel.pricePerRoom * roomCount * hotelNights : 0
    const subtotal = totalTripCost + vehicleCost + hotelCost
    const gst = Math.round(subtotal * 0.18)
    const total = subtotal + gst

    return { tripCostPerHead, totalTripCost, vehicleCost, hotelCost, hotelNights, subtotal, gst, total }
  }

  // Add customer
  const addCustomer = () => {
    setCustomers([
      ...customers,
      {
        id: Date.now().toString(),
        name: "",
        age: "",
        gender: "",
        phone: "",
        email: "",
      },
    ])
  }

  // Remove customer
  const removeCustomer = (id: string) => {
    if (customers.length > 1) {
      setCustomers(customers.filter((c) => c.id !== id))
    }
  }

  // Update customer
  const updateCustomer = (id: string, field: keyof Customer, value: string) => {
    setCustomers(customers.map((c) => (c.id === id ? { ...c, [field]: value } : c)))
  }

  // Check if step is complete
  const isStepComplete = (step: number) => {
    switch (step) {
      case 1:
        return selectedCity !== ""
      case 2:
        return tripDays > 0
      case 3:
        return selectedVehicle !== ""
      case 4:
        return selectedHotel !== ""
      case 5:
        return customers.every((c) => c.name && c.age && c.gender)
      default:
        return false
    }
  }

  // Handle booking confirmation
  const handleConfirmBooking = () => {
    alert("Spiritual package booking confirmed successfully!")
    router.push("/agent/all-bookings")
  }

  if (isLoading) {
    return (
      <div className="flex h-screen items-center justify-center">
        <div className="animate-spin rounded-full h-8 w-8 border-b-2 border-gray-700" />
      </div>
    )
  }

  const pricing = calculatePricing()
  const itinerary = getItinerary()
  const hotels = getHotels()

  return (
    <div className="min-h-screen bg-gradient-to-br from-gray-50 via-white to-gray-100">
      <AgentHeader agentEmail={agentEmail} agentName={agentName} />

      <main className="container mx-auto p-4 md:p-6 lg:p-8">
        {/* Header */}
        <div className="relative overflow-hidden rounded-2xl bg-gradient-to-r from-gray-900 via-gray-800 to-zinc-900 p-6 md:p-8 mb-8">
          <div className="absolute inset-0 bg-[url('data:image/svg+xml;base64,PHN2ZyB3aWR0aD0iNjAiIGhlaWdodD0iNjAiIHZpZXdCb3g9IjAgMCA2MCA2MCIgeG1sbnM9Imh0dHA6Ly93d3cudzMub3JnLzIwMDAvc3ZnIj48ZyBmaWxsPSJub25lIiBmaWxsLXJ1bGU9ImV2ZW5vZGQiPjxnIGZpbGw9IiNmZmYiIGZpbGwtb3BhY2l0eT0iMC4wNSI+PGNpcmNsZSBjeD0iMzAiIGN5PSIzMCIgcj0iMiIvPjwvZz48L2c+PC9zdmc+')] opacity-30" />
          <div className="relative flex items-center gap-4">
            <div className="p-3 rounded-xl bg-white/10 backdrop-blur-sm">
              <Compass className="h-8 w-8 text-white" />
            </div>
            <div>
              <h1 className="text-2xl md:text-3xl font-bold text-white">Spiritual Package Booking</h1>
              <p className="text-gray-300 mt-1">Book spiritual journey packages for your clients</p>
            </div>
          </div>
        </div>

        {/* Steps Progress */}
        <div className="mb-8">
          <div className="flex items-center justify-between overflow-x-auto pb-2">
            {steps.map((step, index) => (
              <div key={step.id} className="flex items-center flex-shrink-0">
                <button
                  onClick={() => {
                    if (step.id < currentStep || isStepComplete(step.id - 1)) {
                      setCurrentStep(step.id)
                    }
                  }}
                  className={cn(
                    "flex items-center gap-2 px-3 py-2 rounded-lg transition-all",
                    currentStep === step.id
                      ? "bg-gradient-to-r from-gray-900 to-gray-700 text-white shadow-lg"
                      : step.id < currentStep
                        ? "bg-gray-200 text-gray-700"
                        : "bg-gray-100 text-gray-400",
                  )}
                >
                  <span
                    className={cn(
                      "flex items-center justify-center w-6 h-6 rounded-full text-xs font-medium",
                      currentStep === step.id
                        ? "bg-white text-gray-900"
                        : step.id < currentStep
                          ? "bg-gray-500 text-white"
                          : "bg-gray-300 text-gray-500",
                    )}
                  >
                    {step.id < currentStep ? <Check className="h-3 w-3" /> : step.id}
                  </span>
                  <span className="hidden sm:inline text-sm font-medium">{step.title}</span>
                </button>
                {index < steps.length - 1 && <ChevronRight className="h-4 w-4 mx-1 text-gray-300 flex-shrink-0" />}
              </div>
            ))}
          </div>
        </div>

        <div className="grid lg:grid-cols-3 gap-6">
          {/* Main Content */}
          <div className="lg:col-span-2 space-y-6">
            {/* Step 1: City Selection (No Category) */}
            {currentStep === 1 && (
              <Card className="border-0 shadow-lg">
                <CardHeader className="border-b bg-gradient-to-r from-gray-50 to-white">
                  <CardTitle className="flex items-center gap-2">
                    <MapPin className="h-5 w-5 text-gray-700" />
                    Select Destination
                  </CardTitle>
                  <CardDescription>Choose your spiritual destination</CardDescription>
                </CardHeader>
                <CardContent className="p-6 space-y-6">
                  <div>
                    <Label className="text-sm font-medium mb-3 block">Select Pilgrimage City</Label>
                    <Select value={selectedCity} onValueChange={setSelectedCity}>
                      <SelectTrigger className="h-12">
                        <SelectValue placeholder="Choose a city" />
                      </SelectTrigger>
                      <SelectContent>
                        {cities.map((city) => (
                          <SelectItem key={city.id} value={city.id}>
                            <span className="font-medium">{city.name}</span>
                            <span className="text-muted-foreground ml-2 text-sm">({city.state})</span>
                          </SelectItem>
                        ))}
                      </SelectContent>
                    </Select>
                  </div>

                  {selectedCity && (
                    <div className="animate-in fade-in slide-in-from-bottom-4 duration-300 p-4 rounded-xl bg-gradient-to-r from-gray-50 to-gray-100 border">
                      <div className="flex items-center gap-2 mb-2">
                        <Star className="h-4 w-4 text-yellow-500" />
                        <span className="font-medium">Spiritual Tour Package</span>
                      </div>
                      <p className="text-sm text-muted-foreground">
                        Experience the divine essence of {cities.find((c) => c.id === selectedCity)?.name} with our
                        curated spiritual tour package including temple visits, aarti ceremonies, and sacred site
                        explorations.
                      </p>
                    </div>
                  )}
                </CardContent>
              </Card>
            )}

            {/* Step 2: Trip Duration with Itinerary Display */}
            {currentStep === 2 && (
              <Card className="border-0 shadow-lg">
                <CardHeader className="border-b bg-gradient-to-r from-gray-50 to-white">
                  <CardTitle className="flex items-center gap-2">
                    <Calendar className="h-5 w-5 text-gray-700" />
                    Select Trip Duration
                  </CardTitle>
                  <CardDescription>Choose how many days you want to travel</CardDescription>
                </CardHeader>
                <CardContent className="p-6 space-y-6">
                  <div>
                    <Label className="text-sm font-medium mb-3 block">Trip Duration</Label>
                    <div className="flex gap-3">
                      {[1, 2, 3, 4].map((days) => (
                        <button
                          key={days}
                          onClick={() => setTripDays(days)}
                          className={cn(
                            "flex-1 py-3 px-4 rounded-xl font-medium transition-all",
                            tripDays === days
                              ? "bg-gradient-to-r from-gray-900 to-gray-700 text-white shadow-lg"
                              : "bg-gray-100 hover:bg-gray-200 text-gray-700",
                          )}
                        >
                          {days} Day{days > 1 ? "s" : ""}
                        </button>
                      ))}
                    </div>
                  </div>

                  <div>
                    <Label className="text-sm font-medium mb-3 block">Your Itinerary Preview</Label>
                    <p className="text-sm text-muted-foreground mb-4">
                      Here&apos;s what your {tripDays}-day spiritual journey includes:
                    </p>
                    <div className="space-y-4">
                      {itinerary.slice(0, tripDays).map((day, index) => (
                        <div
                          key={day.day}
                          className="flex gap-4 p-4 rounded-xl border border-gray-200 bg-white animate-in fade-in slide-in-from-left duration-300"
                          style={{ animationDelay: `${index * 100}ms` }}
                        >
                          <img
                            src={day.image || "/placeholder.svg"}
                            alt={day.title}
                            className="w-32 h-24 object-cover rounded-lg flex-shrink-0"
                          />
                          <div className="flex-1 min-w-0">
                            <div className="flex items-center gap-2 mb-1">
                              <Badge className="bg-gradient-to-r from-gray-900 to-gray-700 text-white">
                                Day {day.day}
                              </Badge>
                            </div>
                            <h4 className="font-semibold text-gray-900">{day.title}</h4>
                            <p className="text-sm text-muted-foreground mt-1">
                              <MapPin className="h-3 w-3 inline mr-1" />
                              {day.destinations.join(", ")}
                            </p>
                            <div className="flex flex-wrap gap-2 mt-2">
                              {day.activities.map((activity, actIdx) => (
                                <span key={actIdx} className="text-xs text-gray-500 flex items-center gap-1">
                                  <Clock className="h-3 w-3" />
                                  {activity.time} - {activity.title}
                                </span>
                              ))}
                            </div>
                          </div>
                        </div>
                      ))}
                    </div>
                  </div>
                </CardContent>
              </Card>
            )}

            {/* Step 3: Vehicle Selection */}
            {currentStep === 3 && (
              <Card className="border-0 shadow-lg">
                <CardHeader className="border-b bg-gradient-to-r from-gray-50 to-white">
                  <CardTitle className="flex items-center gap-2">
                    <Car className="h-5 w-5 text-gray-700" />
                    Select Vehicle
                  </CardTitle>
                  <CardDescription>Choose transportation for your group</CardDescription>
                </CardHeader>
                <CardContent className="p-6 space-y-6">
                  <div>
                    <Label className="text-sm font-medium mb-3 block">Number of Passengers</Label>
                    <Select
                      value={passengerCount.toString()}
                      onValueChange={(v) => {
                        setPassengerCount(Number.parseInt(v))
                        setSelectedVehicle("")
                      }}
                    >
                      <SelectTrigger className="h-12 w-full md:w-48">
                        <SelectValue />
                      </SelectTrigger>
                      <SelectContent>
                        {Array.from({ length: 20 }, (_, i) => i + 1).map((num) => (
                          <SelectItem key={num} value={num.toString()}>
                            {num} Passenger{num > 1 ? "s" : ""}
                          </SelectItem>
                        ))}
                      </SelectContent>
                    </Select>
                  </div>

                  <div className="grid md:grid-cols-2 gap-4">
                    {filteredVehicles.map((vehicle) => (
                      <button
                        key={vehicle.id}
                        onClick={() => setSelectedVehicle(vehicle.id)}
                        className={cn(
                          "p-4 rounded-xl border-2 text-left transition-all",
                          selectedVehicle === vehicle.id
                            ? "border-gray-900 bg-gray-50 shadow-lg"
                            : "border-gray-200 hover:border-gray-400 hover:shadow",
                        )}
                      >
                        <div className="flex gap-4">
                          <img
                            src={vehicle.image || "/placeholder.svg"}
                            alt={vehicle.name}
                            className="w-24 h-16 object-cover rounded-lg bg-gray-100"
                          />
                          <div className="flex-1">
                            <h4 className="font-semibold text-gray-900">{vehicle.name}</h4>
                            <div className="flex items-center gap-2 mt-1 text-sm text-gray-600">
                              <Users className="h-3 w-3" />
                              <span>{vehicle.seats} Seats</span>
                            </div>
                            <div className="flex flex-wrap gap-1 mt-2">
                              {vehicle.features.map((feature) => (
                                <Badge key={feature} variant="secondary" className="text-xs">
                                  {feature}
                                </Badge>
                              ))}
                            </div>
                          </div>
                        </div>
                        <div className="mt-3 pt-3 border-t flex items-center justify-between">
                          <span className="text-sm text-gray-500">Per Day</span>
                          <span className="font-bold text-lg flex items-center">
                            <IndianRupee className="h-4 w-4" />
                            {vehicle.pricePerDay.toLocaleString()}
                          </span>
                        </div>
                      </button>
                    ))}
                  </div>

                  {filteredVehicles.length === 0 && (
                    <div className="text-center py-8 text-gray-500">
                      <Car className="h-12 w-12 mx-auto mb-3 opacity-50" />
                      <p>No vehicles available for {passengerCount} passengers</p>
                    </div>
                  )}
                </CardContent>
              </Card>
            )}

            {/* Step 4: Hotel Selection (Pre-defined hotels, not star category) */}
            {currentStep === 4 && (
              <Card className="border-0 shadow-lg">
                <CardHeader className="border-b bg-gradient-to-r from-gray-50 to-white">
                  <CardTitle className="flex items-center gap-2">
                    <Hotel className="h-5 w-5 text-gray-700" />
                    Select Hotel
                  </CardTitle>
                  <CardDescription>Choose your accommodation from available hotels</CardDescription>
                </CardHeader>
                <CardContent className="p-6 space-y-6">
                  <div>
                    <Label className="text-sm font-medium mb-3 block">Number of Rooms</Label>
                    <Select value={roomCount.toString()} onValueChange={(v) => setRoomCount(Number.parseInt(v))}>
                      <SelectTrigger className="h-12 w-full md:w-48">
                        <SelectValue />
                      </SelectTrigger>
                      <SelectContent>
                        {Array.from({ length: 10 }, (_, i) => i + 1).map((num) => (
                          <SelectItem key={num} value={num.toString()}>
                            {num} Room{num > 1 ? "s" : ""}
                          </SelectItem>
                        ))}
                      </SelectContent>
                    </Select>
                    {tripDays > 1 && (
                      <p className="text-sm text-muted-foreground mt-2">
                        Stay: {tripDays - 1} Night{tripDays - 1 > 1 ? "s" : ""}
                      </p>
                    )}
                  </div>

                  <div className="grid md:grid-cols-2 gap-4">
                    {hotels.map((hotel) => (
                      <button
                        key={hotel.id}
                        onClick={() => setSelectedHotel(hotel.id)}
                        className={cn(
                          "p-4 rounded-xl border-2 text-left transition-all",
                          selectedHotel === hotel.id
                            ? "border-gray-900 bg-gray-50 shadow-lg"
                            : "border-gray-200 hover:border-gray-400 hover:shadow",
                        )}
                      >
                        <img
                          src={hotel.image || "/placeholder.svg"}
                          alt={hotel.name}
                          className="w-full h-32 object-cover rounded-lg mb-3"
                        />
                        <div className="flex items-start justify-between mb-2">
                          <div>
                            <h4 className="font-semibold text-gray-900">{hotel.name}</h4>
                            <p className="text-sm text-muted-foreground flex items-center gap-1">
                              <MapPin className="h-3 w-3" />
                              {hotel.location}
                            </p>
                          </div>
                          <div className="flex items-center gap-1">
                            <Star className="h-4 w-4 fill-yellow-400 text-yellow-400" />
                            <span className="font-medium">{hotel.rating}</span>
                          </div>
                        </div>
                        <div className="flex flex-wrap gap-1 mb-3">
                          {hotel.amenities.slice(0, 3).map((amenity) => (
                            <Badge key={amenity} variant="secondary" className="text-xs">
                              {amenity}
                            </Badge>
                          ))}
                        </div>
                        <div className="pt-3 border-t flex items-center justify-between">
                          <span className="text-sm text-gray-500">Per Night</span>
                          <span className="font-bold text-lg flex items-center">
                            <IndianRupee className="h-4 w-4" />
                            {hotel.pricePerRoom.toLocaleString()}
                          </span>
                        </div>
                      </button>
                    ))}
                  </div>
                </CardContent>
              </Card>
            )}

            {/* Step 5: Customer Details */}
            {currentStep === 5 && (
              <Card className="border-0 shadow-lg">
                <CardHeader className="border-b bg-gradient-to-r from-gray-50 to-white">
                  <CardTitle className="flex items-center gap-2">
                    <Users className="h-5 w-5 text-gray-700" />
                    Customer Details
                  </CardTitle>
                  <CardDescription>Enter details for all travelers</CardDescription>
                </CardHeader>
                <CardContent className="p-6 space-y-6">
                  {customers.map((customer, index) => (
                    <div
                      key={customer.id}
                      className="p-4 rounded-xl border border-gray-200 bg-white animate-in fade-in duration-300"
                    >
                      <div className="flex items-center justify-between mb-4">
                        <h4 className="font-semibold flex items-center gap-2">
                          <User className="h-4 w-4" />
                          {index === 0 ? "Primary Traveler" : `Traveler ${index + 1}`}
                        </h4>
                        {customers.length > 1 && (
                          <Button
                            variant="ghost"
                            size="sm"
                            onClick={() => removeCustomer(customer.id)}
                            className="text-red-500 hover:text-red-600 hover:bg-red-50"
                          >
                            <Trash2 className="h-4 w-4" />
                          </Button>
                        )}
                      </div>
                      <div className="grid md:grid-cols-3 gap-4">
                        <div>
                          <Label className="text-sm">Full Name *</Label>
                          <Input
                            value={customer.name}
                            onChange={(e) => updateCustomer(customer.id, "name", e.target.value)}
                            placeholder="Enter full name"
                            className="mt-1"
                          />
                        </div>
                        <div>
                          <Label className="text-sm">Age *</Label>
                          <Input
                            type="number"
                            value={customer.age}
                            onChange={(e) => updateCustomer(customer.id, "age", e.target.value)}
                            placeholder="Age"
                            className="mt-1"
                          />
                        </div>
                        <div>
                          <Label className="text-sm">Gender *</Label>
                          <Select
                            value={customer.gender}
                            onValueChange={(v) => updateCustomer(customer.id, "gender", v)}
                          >
                            <SelectTrigger className="mt-1">
                              <SelectValue placeholder="Select" />
                            </SelectTrigger>
                            <SelectContent>
                              <SelectItem value="male">Male</SelectItem>
                              <SelectItem value="female">Female</SelectItem>
                              <SelectItem value="other">Other</SelectItem>
                            </SelectContent>
                          </Select>
                        </div>
                        {index === 0 && (
                          <>
                            <div>
                              <Label className="text-sm">Phone Number</Label>
                              <div className="relative mt-1">
                                <Phone className="absolute left-3 top-1/2 -translate-y-1/2 h-4 w-4 text-gray-400" />
                                <Input
                                  value={customer.phone}
                                  onChange={(e) => updateCustomer(customer.id, "phone", e.target.value)}
                                  placeholder="+91 98765 43210"
                                  className="pl-10"
                                />
                              </div>
                            </div>
                            <div className="md:col-span-2">
                              <Label className="text-sm">Email Address</Label>
                              <div className="relative mt-1">
                                <Mail className="absolute left-3 top-1/2 -translate-y-1/2 h-4 w-4 text-gray-400" />
                                <Input
                                  type="email"
                                  value={customer.email}
                                  onChange={(e) => updateCustomer(customer.id, "email", e.target.value)}
                                  placeholder="email@example.com"
                                  className="pl-10"
                                />
                              </div>
                            </div>
                          </>
                        )}
                      </div>
                    </div>
                  ))}
                  <Button
                    variant="outline"
                    onClick={addCustomer}
                    className="w-full border-dashed border-2 hover:bg-gray-50 bg-transparent"
                  >
                    <Plus className="h-4 w-4 mr-2" />
                    Add Another Traveler
                  </Button>
                </CardContent>
              </Card>
            )}

            {/* Step 6: Review & Confirm */}
            {currentStep === 6 && (
              <Card className="border-0 shadow-lg">
                <CardHeader className="border-b bg-gradient-to-r from-gray-50 to-white">
                  <CardTitle className="flex items-center gap-2">
                    <CheckCircle2 className="h-5 w-5 text-gray-700" />
                    Review & Confirm
                  </CardTitle>
                  <CardDescription>Review your booking details before confirming</CardDescription>
                </CardHeader>
                <CardContent className="p-6 space-y-6">
                  <div className="space-y-4">
                    <div className="p-4 rounded-xl bg-gray-50 border">
                      <h4 className="font-semibold flex items-center gap-2 mb-3">
                        <MapPin className="h-4 w-4" />
                        Trip Details
                      </h4>
                      <div className="grid grid-cols-2 gap-4 text-sm">
                        <div>
                          <p className="text-muted-foreground">Destination</p>
                          <p className="font-medium">
                            {cities.find((c) => c.id === selectedCity)?.name || selectedCity}
                          </p>
                        </div>
                        <div>
                          <p className="text-muted-foreground">Tour Type</p>
                          <p className="font-medium">Spiritual Package</p>
                        </div>
                        <div>
                          <p className="text-muted-foreground">Duration</p>
                          <p className="font-medium">
                            {tripDays} Day{tripDays > 1 ? "s" : ""}
                          </p>
                        </div>
                        <div>
                          <p className="text-muted-foreground">Travelers</p>
                          <p className="font-medium">{customers.length} Person(s)</p>
                        </div>
                      </div>
                    </div>

                    <div className="p-4 rounded-xl bg-gray-50 border">
                      <h4 className="font-semibold flex items-center gap-2 mb-3">
                        <Car className="h-4 w-4" />
                        Vehicle
                      </h4>
                      <p className="font-medium">{vehicles.find((v) => v.id === selectedVehicle)?.name}</p>
                      <p className="text-sm text-muted-foreground">
                        {tripDays} days x Rs.{" "}
                        {vehicles.find((v) => v.id === selectedVehicle)?.pricePerDay.toLocaleString()}
                      </p>
                    </div>

                    <div className="p-4 rounded-xl bg-gray-50 border">
                      <h4 className="font-semibold flex items-center gap-2 mb-3">
                        <Hotel className="h-4 w-4" />
                        Accommodation
                      </h4>
                      <p className="font-medium">{hotels.find((h) => h.id === selectedHotel)?.name}</p>
                      <p className="text-sm text-muted-foreground">
                        {roomCount} room(s) x {pricing.hotelNights} night(s)
                      </p>
                    </div>

                    <div className="p-4 rounded-xl bg-gray-50 border">
                      <h4 className="font-semibold flex items-center gap-2 mb-3">
                        <Users className="h-4 w-4" />
                        Travelers
                      </h4>
                      <div className="space-y-2">
                        {customers.map((customer) => (
                          <div key={customer.id} className="text-sm">
                            <span className="font-medium">{customer.name}</span>
                            <span className="text-muted-foreground ml-2">
                              ({customer.age} yrs, {customer.gender})
                            </span>
                          </div>
                        ))}
                      </div>
                    </div>
                  </div>
                </CardContent>
              </Card>
            )}
          </div>

          {/* Sidebar - Price Summary */}
          <div className="lg:col-span-1">
            <div className="sticky top-4">
              <Card className="border-0 shadow-lg overflow-hidden">
                <CardHeader className="bg-gradient-to-r from-gray-900 via-gray-800 to-zinc-900 text-white">
                  <CardTitle className="flex items-center gap-2">
                    <IndianRupee className="h-5 w-5" />
                    Price Summary
                  </CardTitle>
                </CardHeader>
                <CardContent className="p-4 space-y-4">
                  {selectedCity ? (
                    <>
                      <div className="space-y-2">
                        <h4 className="font-medium text-sm text-gray-700">Your Selection</h4>
                        <div className="space-y-1 text-sm">
                          <div className="flex items-center gap-2">
                            <MapPin className="h-3 w-3 text-gray-400" />
                            <span>{cities.find((c) => c.id === selectedCity)?.name}</span>
                          </div>
                          <div className="flex items-center gap-2">
                            <Star className="h-3 w-3 text-gray-400" />
                            <span>Spiritual Package</span>
                          </div>
                          <div className="flex items-center gap-2">
                            <Calendar className="h-3 w-3 text-gray-400" />
                            <span>
                              {tripDays} Day{tripDays > 1 ? "s" : ""}
                            </span>
                          </div>
                          {selectedVehicle && (
                            <div className="flex items-center gap-2">
                              <Car className="h-3 w-3 text-gray-400" />
                              <span>{vehicles.find((v) => v.id === selectedVehicle)?.name}</span>
                            </div>
                          )}
                          {selectedHotel && (
                            <div className="flex items-center gap-2">
                              <Hotel className="h-3 w-3 text-gray-400" />
                              <span>{hotels.find((h) => h.id === selectedHotel)?.name}</span>
                            </div>
                          )}
                          {customers.length > 0 && (
                            <div className="flex items-center gap-2">
                              <Users className="h-3 w-3 text-gray-400" />
                              <span>{customers.length} Traveler(s)</span>
                            </div>
                          )}
                        </div>
                      </div>

                      <hr />

                      <div className="space-y-2 text-sm">
                        <div className="flex justify-between">
                          <span className="text-gray-600">
                            Trip Cost ({customers.length} x Rs. {pricing.tripCostPerHead.toLocaleString()})
                          </span>
                          <span className="font-medium flex items-center">
                            <IndianRupee className="h-3 w-3" />
                            {pricing.totalTripCost.toLocaleString()}
                          </span>
                        </div>
                        {selectedVehicle && (
                          <div className="flex justify-between">
                            <span className="text-gray-600">Vehicle ({tripDays} days)</span>
                            <span className="font-medium flex items-center">
                              <IndianRupee className="h-3 w-3" />
                              {pricing.vehicleCost.toLocaleString()}
                            </span>
                          </div>
                        )}
                        {selectedHotel && pricing.hotelNights > 0 && (
                          <div className="flex justify-between">
                            <span className="text-gray-600">
                              Hotel ({roomCount} room x {pricing.hotelNights} nights)
                            </span>
                            <span className="font-medium flex items-center">
                              <IndianRupee className="h-3 w-3" />
                              {pricing.hotelCost.toLocaleString()}
                            </span>
                          </div>
                        )}
                        <hr />
                        <div className="flex justify-between">
                          <span className="text-gray-600">Subtotal</span>
                          <span className="font-medium flex items-center">
                            <IndianRupee className="h-3 w-3" />
                            {pricing.subtotal.toLocaleString()}
                          </span>
                        </div>
                        <div className="flex justify-between">
                          <span className="text-gray-600">GST (18%)</span>
                          <span className="font-medium flex items-center">
                            <IndianRupee className="h-3 w-3" />
                            {pricing.gst.toLocaleString()}
                          </span>
                        </div>
                        <hr />
                        <div className="flex justify-between text-lg font-bold">
                          <span>Total</span>
                          <span className="flex items-center text-gray-900">
                            <IndianRupee className="h-4 w-4" />
                            {pricing.total.toLocaleString()}
                          </span>
                        </div>
                      </div>
                    </>
                  ) : (
                    <p className="text-sm text-muted-foreground text-center py-4">Complete selections to see pricing</p>
                  )}
                </CardContent>
              </Card>

              <div className="flex gap-3 mt-4">
                {currentStep > 1 && (
                  <Button
                    variant="outline"
                    className="flex-1 bg-transparent"
                    onClick={() => setCurrentStep(currentStep - 1)}
                  >
                    Back
                  </Button>
                )}
                {currentStep < 6 ? (
                  <Button
                    className="flex-1 bg-gradient-to-r from-gray-900 to-gray-700 hover:from-gray-800 hover:to-gray-600"
                    disabled={!isStepComplete(currentStep)}
                    onClick={() => setCurrentStep(currentStep + 1)}
                  >
                    Continue
                    <ChevronRight className="h-4 w-4 ml-1" />
                  </Button>
                ) : (
                  <Button
                    className="flex-1 bg-gradient-to-r from-gray-900 to-gray-700 hover:from-gray-800 hover:to-gray-600"
                    onClick={handleConfirmBooking}
                  >
                    <Check className="h-4 w-4 mr-2" />
                    Confirm Booking
                  </Button>
                )}
              </div>
            </div>
          </div>
        </div>
      </main>
    </div>
  )
}
