import type React from "react"
import type { Metadata, Viewport } from "next"
import { Geist, Geist_Mono } from "next/font/google"
import { Analytics } from "@vercel/analytics/next"
import { Toaster } from "@/components/ui/sonner"
import "./globals.css"

const _geist = Geist({ subsets: ["latin"] })
const _geistMono = Geist_Mono({ subsets: ["latin"] })

export const metadata: Metadata = {
  metadataBase: new URL(process.env.NEXT_PUBLIC_SITE_URL || "https://pnpplatform.com"),
  title: {
    default: "PNP Platform - Your Trusted Partner",
    template: "%s | PNP Platform",
  },
  description: "PNP Platform provides comprehensive solutions and innovative services for your business needs.",
  keywords: ["PNP Platform", "business solutions", "innovation", "services"],
  authors: [{ name: "PNP Platform" }],
  creator: "PNP Platform",
  publisher: "PNP Platform",
  formatDetection: {
    email: false,
    address: false,
    telephone: false,
  },
  openGraph: {
    type: "website",
    locale: "en_US",
    url: process.env.NEXT_PUBLIC_SITE_URL || "https://pnpplatform.com",
    siteName: "PNP Platform",
    title: "PNP Platform - Your Trusted Partner",
    description: "PNP Platform provides comprehensive solutions and innovative services for your business needs.",
    images: [
      {
        url: "/og-image.jpg",
        width: 1200,
        height: 630,
        alt: "PNP Platform",
      },
    ],
  },
  twitter: {
    card: "summary_large_image",
    title: "PNP Platform - Your Trusted Partner",
    description: "PNP Platform provides comprehensive solutions and innovative services for your business needs.",
    images: ["/og-image.jpg"],
  },
  robots: {
    index: true,
    follow: true,
    googleBot: {
      index: true,
      follow: true,
      "max-video-preview": -1,
      "max-image-preview": "large",
      "max-snippet": -1,
    },
  },
  icons: {
    icon: [
      {
        url: "/icon-light-32x32.png",
        media: "(prefers-color-scheme: light)",
      },
      {
        url: "/icon-dark-32x32.png",
        media: "(prefers-color-scheme: dark)",
      },
      {
        url: "/icon.svg",
        type: "image/svg+xml",
      },
    ],
    apple: "/apple-icon.png",
  },
  verification: {
    google: "your-google-verification-code",
  },
  generator: "v0.app",
}

export const viewport: Viewport = {
  themeColor: [
    { media: "(prefers-color-scheme: light)", color: "white" },
    { media: "(prefers-color-scheme: dark)", color: "black" },
  ],
  width: "device-width",
  initialScale: 1,
  maximumScale: 5,
  userScalable: true,
}

export default function RootLayout({
  children,
}: Readonly<{
  children: React.ReactNode
}>) {
  return (
    <html lang="en" suppressHydrationWarning>
      <body className={`font-sans antialiased bg-background`}>
        {children}
        <Toaster richColors closeButton />
        <Analytics />
      </body>
    </html>
  )
}
