"use client"

import { useState } from "react"
import { cn } from "@/lib/utils"
import Link from "next/link"
import { usePathname } from "next/navigation"
import {
  LayoutDashboard,
  UserCog,
  ChevronDown,
  Users,
  Shield,
  UsersRound,
  Briefcase,
  Hotel,
  Building2,
  CalendarCheck,
  Car,
  Route,
  CarFront,
  Package,
  Sparkles,
  Settings2,
  PackageCheck,
  ClipboardList,
  DollarSign,
  Wallet,
  MapPin,
  Map,
  FolderTree,
  Receipt,
  TrendingUp,
  Scale,
  FileText,
  ShieldCheck,
  Ban,
  RefreshCcw,
  HelpCircle,
  UserCheck,
  Building,
  CreditCard,
} from "lucide-react"
import { Collapsible, CollapsibleContent, CollapsibleTrigger } from "@/components/ui/collapsible"

const navigation = [{ name: "Dashboard", href: "/admin/dashboard", icon: LayoutDashboard }]

const topDropdownMenus = [
  {
    id: "users",
    name: "Users",
    icon: Users,
    children: [
      { name: "Customers", href: "/admin/customers", icon: UsersRound },
      { name: "Agents", href: "/admin/agents", icon: Briefcase },
    ],
  },
  {
    id: "category",
    name: "Category",
    icon: FolderTree,
    children: [
      { name: "States", href: "/admin/states", icon: Map },
      { name: "Cities", href: "/admin/cities", icon: MapPin },
    ],
  },
]

const dropdownMenus = [
  {
    id: "hotel-bookings",
    name: "Hotel & Bookings",
    icon: Hotel,
    children: [
      { name: "Hotel Partners", href: "/admin/hotel-partners", icon: Building2 },
      { name: "Hotels Booking", href: "/admin/hotels-booking", icon: CalendarCheck },
    ],
  },
  {
    id: "vehicle-bookings",
    name: "Vehicle & Bookings",
    icon: Car,
    children: [
      { name: "Vehicle Listing", href: "/admin/vehicle-listing", icon: CarFront },
      { name: "Routes Listing", href: "/admin/routes-listing", icon: Route },
      { name: "Vehicle Bookings", href: "/admin/vehicle-bookings", icon: CalendarCheck },
    ],
  },
  {
    id: "packages-control",
    name: "Packages & Control",
    icon: Package,
    children: [
      { name: "Fix Packages", href: "/admin/fix-packages", icon: PackageCheck },
      { name: "Spiritual Packages", href: "/admin/spiritual-packages", icon: Sparkles },
      { name: "Custom Packages", href: "/admin/custom-packages", icon: Settings2 },
    ],
  },
  {
    id: "packages-bookings",
    name: "Packages Bookings",
    icon: ClipboardList,
    children: [
      { name: "Fix Packages Bookings", href: "/admin/fix-packages-bookings", icon: PackageCheck },
      { name: "Spiritual Packages Booking", href: "/admin/spiritual-packages-bookings", icon: Sparkles },
      { name: "Custom Packages Booking", href: "/admin/custom-packages-bookings", icon: Settings2 },
    ],
  },
  {
    id: "agent-hotel-bookings",
    name: "Agent Bookings",
    icon: Receipt,
    children: [
      { name: "Agent Bookings", href: "/admin/agent-booking", icon: Briefcase },
      { name: "Hotel Bookings", href: "/admin/hotel-booking", icon: Hotel },
    ],
  },
  {
    id: "earnings-reports",
    name: "Earnings Reports",
    icon: TrendingUp,
    children: [
      { name: "Agent Earning", href: "/admin/agent-earning", icon: DollarSign },
      { name: "Hotel Earning", href: "/admin/hotel-earning", icon: DollarSign },
    ],
  },
  {
    id: "payment-withdraw",
    name: "Payment Withdraw",
    icon: Wallet,
    children: [
      { name: "Agent Withdraw", href: "/admin/agent-withdraw", icon: Wallet },
      { name: "Hotel Withdraw", href: "/admin/hotel-withdraw", icon: Wallet },
    ],
  },
]

const adminRolesMenu = {
  id: "admin-roles",
  name: "Admin & Roles",
  icon: UserCog,
  children: [
    { name: "Manage Team", href: "/admin/sub-admin/manage-team", icon: Users },
    { name: "Manage Roles", href: "/admin/sub-admin/manage-roles", icon: Shield },
  ],
}

const legalTermsMenu = {
  id: "legal-terms",
  name: "Legal & Terms",
  icon: Scale,
  children: [
    { name: "User Terms", href: "/admin/user-terms", icon: FileText },
    { name: "User Privacy", href: "/admin/privacy-policy", icon: ShieldCheck },
    { name: "User Cancellation Policy", href: "/admin/cancellation-policy", icon: Ban },
    { name: "User Refund Policy", href: "/admin/user-refund-policy", icon: RefreshCcw },
    { name: "FAQ", href: "/admin/user-faq", icon: HelpCircle },
    { name: "Agent Terms", href: "/admin/agent-terms", icon: UserCheck },
    { name: "Hotel Terms", href: "/admin/hotel-terms", icon: Building },
    { name: "Withdraw Terms", href: "/admin/withdraw-terms", icon: CreditCard },
  ],
}

interface AdminSidebarNavProps {
  onNavigate?: () => void
}

export function AdminSidebarNav({ onNavigate }: AdminSidebarNavProps) {
  const pathname = usePathname()

  const [openMenus, setOpenMenus] = useState<Record<string, boolean>>(() => {
    const initial: Record<string, boolean> = {}
    topDropdownMenus.forEach((menu) => {
      initial[menu.id] = menu.children.some((child) => pathname.startsWith(child.href))
    })
    dropdownMenus.forEach((menu) => {
      initial[menu.id] = menu.children.some((child) => pathname.startsWith(child.href))
    })
    initial["admin-roles"] = pathname.startsWith("/admin/sub-admin")
    initial["legal-terms"] = legalTermsMenu.children.some((child) => pathname.startsWith(child.href))
    return initial
  })

  const toggleMenu = (menuId: string) => {
    setOpenMenus((prev) => {
      const isCurrentlyOpen = prev[menuId]
      // If we're opening this menu, close all others first
      if (!isCurrentlyOpen) {
        const allClosed: Record<string, boolean> = {}
        Object.keys(prev).forEach((key) => {
          allClosed[key] = false
        })
        return { ...allClosed, [menuId]: true }
      }
      // If we're closing this menu, just toggle it
      return { ...prev, [menuId]: false }
    })
  }

  const isMenuActive = (children: { href: string }[]) => {
    return children.some((child) => pathname === child.href || pathname.startsWith(child.href + "/"))
  }

  const renderDropdownMenu = (menu: (typeof dropdownMenus)[0]) => {
    const isActive = isMenuActive(menu.children)
    const MenuIcon = menu.icon

    return (
      <Collapsible key={menu.id} open={openMenus[menu.id]} onOpenChange={() => toggleMenu(menu.id)}>
        <CollapsibleTrigger
          className={cn(
            "group flex w-full items-center justify-between gap-3 rounded-lg px-3 py-2.5 text-sm font-medium transition-all duration-200",
            "focus-visible:outline-none focus-visible:ring-2 focus-visible:ring-ring focus-visible:ring-offset-2",
            isActive ? "bg-primary/10 text-primary" : "text-muted-foreground hover:bg-accent/50 hover:text-foreground",
          )}
          aria-label={`Toggle ${menu.name} menu`}
        >
          <span className="flex items-center gap-3">
            <MenuIcon
              className={cn(
                "h-4 w-4 transition-colors",
                isActive ? "text-primary" : "text-muted-foreground group-hover:text-foreground",
              )}
              aria-hidden="true"
            />
            {menu.name}
          </span>
          <ChevronDown
            className={cn(
              "h-4 w-4 text-muted-foreground transition-transform duration-200",
              openMenus[menu.id] && "rotate-180",
            )}
            aria-hidden="true"
          />
        </CollapsibleTrigger>
        <CollapsibleContent className="mt-0.5 space-y-0.5 pl-3">
          {menu.children.map((child) => {
            const isChildActive = pathname === child.href || pathname.startsWith(child.href + "/")
            const ChildIcon = child.icon

            return (
              <Link
                key={child.name}
                href={child.href}
                onClick={onNavigate}
                className={cn(
                  "group flex items-center gap-3 rounded-lg px-3 py-2 text-sm font-medium transition-all duration-200",
                  "focus-visible:outline-none focus-visible:ring-2 focus-visible:ring-ring focus-visible:ring-offset-2",
                  isChildActive
                    ? "bg-primary text-primary-foreground shadow-sm"
                    : "text-muted-foreground hover:bg-accent/50 hover:text-foreground",
                )}
                aria-current={isChildActive ? "page" : undefined}
              >
                <ChildIcon
                  className={cn(
                    "h-4 w-4 transition-colors",
                    isChildActive ? "text-primary-foreground" : "text-muted-foreground group-hover:text-foreground",
                  )}
                  aria-hidden="true"
                />
                {child.name}
              </Link>
            )
          })}
        </CollapsibleContent>
      </Collapsible>
    )
  }

  return (
    <nav className="flex flex-col space-y-1">
      {/* Dashboard Link */}
      {navigation.map((item) => {
        const isActive =
          pathname === item.href || (item.href !== "/admin/dashboard" && pathname.startsWith(item.href + "/"))
        const Icon = item.icon

        return (
          <Link
            key={item.name}
            href={item.href}
            onClick={onNavigate}
            className={cn(
              "group flex items-center gap-3 rounded-lg px-3 py-2.5 text-sm font-medium transition-all duration-200",
              "focus-visible:outline-none focus-visible:ring-2 focus-visible:ring-ring focus-visible:ring-offset-2",
              isActive
                ? "bg-primary text-primary-foreground shadow-sm"
                : "text-muted-foreground hover:bg-accent/50 hover:text-foreground",
            )}
            aria-current={isActive ? "page" : undefined}
          >
            <Icon
              className={cn(
                "h-4 w-4 transition-colors",
                isActive ? "text-primary-foreground" : "text-muted-foreground group-hover:text-foreground",
              )}
              aria-hidden="true"
            />
            {item.name}
          </Link>
        )
      })}

      {topDropdownMenus.map(renderDropdownMenu)}
      {dropdownMenus.map(renderDropdownMenu)}
      {renderDropdownMenu(adminRolesMenu)}
      {renderDropdownMenu(legalTermsMenu)}
    </nav>
  )
}
