"use client"

import { useState, useMemo } from "react"
import { Tabs, TabsContent, TabsList, TabsTrigger } from "@/components/ui/tabs"
import { AgentTable } from "./agent-table"
import { AgentDetailsDialog } from "./agent-details-dialog"
import { BlockAgentDialog } from "./block-agent-dialog"
import { RejectAgentDialog } from "./reject-agent-dialog"
import { agentDataStore, type Agent } from "@/lib/data/agents"
import { useToastEnhanced } from "@/hooks/use-toast-enhanced"

export function AgentsClient() {
  const toast = useToastEnhanced()
  const [agents, setAgents] = useState<Agent[]>(agentDataStore.getAllAgents())
  const [selectedAgent, setSelectedAgent] = useState<Agent | null>(null)
  const [isDetailsOpen, setIsDetailsOpen] = useState(false)
  const [isBlockDialogOpen, setIsBlockDialogOpen] = useState(false)
  const [isRejectDialogOpen, setIsRejectDialogOpen] = useState(false)
  const [agentToBlock, setAgentToBlock] = useState<Agent | null>(null)
  const [agentToReject, setAgentToReject] = useState<Agent | null>(null)

  const pendingAgents = useMemo(() => agents.filter((a) => a.status === "pending"), [agents])
  const approvedAgents = useMemo(() => agents.filter((a) => a.status === "approved"), [agents])
  const rejectedAgents = useMemo(() => agents.filter((a) => a.status === "rejected"), [agents])
  const blockedAgents = useMemo(() => agents.filter((a) => a.status === "blocked"), [agents])

  const handleViewDetails = (agent: Agent) => {
    setSelectedAgent(agent)
    setIsDetailsOpen(true)
  }

  const handleApprove = (agent: Agent) => {
    const updated = agentDataStore.approveAgent(agent.id)
    if (updated) {
      setAgents(agentDataStore.getAllAgents())
      toast.success({
        title: "Agent Approved",
        description: `${agent.firstName} ${agent.lastName} has been approved as an agent.`,
      })
    }
  }

  const handleRejectClick = (agent: Agent) => {
    setAgentToReject(agent)
    setIsRejectDialogOpen(true)
  }

  const handleRejectConfirm = (reason: string) => {
    if (!agentToReject) return

    const updated = agentDataStore.rejectAgent(agentToReject.id, reason)
    if (updated) {
      setAgents(agentDataStore.getAllAgents())
      toast.error({
        title: "Agent Rejected",
        description: `${agentToReject.firstName} ${agentToReject.lastName} has been rejected.`,
      })
    }
    setIsRejectDialogOpen(false)
    setAgentToReject(null)
  }

  const handleBlockClick = (agent: Agent) => {
    setAgentToBlock(agent)
    setIsBlockDialogOpen(true)
  }

  const handleBlockConfirm = (reason: string) => {
    if (!agentToBlock) return

    const updated = agentDataStore.blockAgent(agentToBlock.id, reason)
    if (updated) {
      setAgents(agentDataStore.getAllAgents())
      toast.error({
        title: "Agent Blocked",
        description: `${agentToBlock.firstName} ${agentToBlock.lastName} has been blocked.`,
      })
    }
    setIsBlockDialogOpen(false)
    setAgentToBlock(null)
  }

  const handleUnblock = (agent: Agent) => {
    const updated = agentDataStore.unblockAgent(agent.id)
    if (updated) {
      setAgents(agentDataStore.getAllAgents())
      toast.success({
        title: "Agent Unblocked",
        description: `${agent.firstName} ${agent.lastName} has been restored to approved status.`,
      })
    }
  }

  const handleRestore = (agent: Agent) => {
    const updated = agentDataStore.restoreRejectedAgent(agent.id)
    if (updated) {
      setAgents(agentDataStore.getAllAgents())
      toast.success({
        title: "Agent Restored",
        description: `${agent.firstName} ${agent.lastName} has been moved to pending for review.`,
      })
    }
  }

  return (
    <section aria-labelledby="agents-heading" className="admin-section">
      <header className="admin-page-header">
        <h1 id="agents-heading" className="admin-page-title">
          Agents
        </h1>
        <p className="admin-page-description">Manage travel agents, review applications, and monitor their activity.</p>
      </header>

      <Tabs defaultValue="all" className="w-full">
        <TabsList className="mb-6 h-auto p-1.5 bg-muted/50 rounded-xl border border-border/50 flex-wrap">
          <TabsTrigger
            value="all"
            className="rounded-lg data-[state=active]:bg-background data-[state=active]:shadow-sm px-4 py-2"
          >
            All ({agents.length})
          </TabsTrigger>
          <TabsTrigger
            value="pending"
            className="rounded-lg data-[state=active]:bg-background data-[state=active]:shadow-sm px-4 py-2"
          >
            Pending ({pendingAgents.length})
          </TabsTrigger>
          <TabsTrigger
            value="approved"
            className="rounded-lg data-[state=active]:bg-background data-[state=active]:shadow-sm px-4 py-2"
          >
            Approved ({approvedAgents.length})
          </TabsTrigger>
          <TabsTrigger
            value="rejected"
            className="rounded-lg data-[state=active]:bg-background data-[state=active]:shadow-sm px-4 py-2"
          >
            Rejected ({rejectedAgents.length})
          </TabsTrigger>
          <TabsTrigger
            value="blocked"
            className="rounded-lg data-[state=active]:bg-background data-[state=active]:shadow-sm px-4 py-2"
          >
            Blocked ({blockedAgents.length})
          </TabsTrigger>
        </TabsList>

        <TabsContent value="all" className="mt-0">
          <AgentTable
            agents={agents}
            onViewDetails={handleViewDetails}
            onApprove={handleApprove}
            onReject={handleRejectClick}
            onBlock={handleBlockClick}
            onUnblock={handleUnblock}
            onRestore={handleRestore}
          />
        </TabsContent>

        <TabsContent value="pending" className="mt-0">
          <AgentTable
            agents={pendingAgents}
            onViewDetails={handleViewDetails}
            onApprove={handleApprove}
            onReject={handleRejectClick}
            onBlock={handleBlockClick}
            onUnblock={handleUnblock}
            onRestore={handleRestore}
          />
        </TabsContent>

        <TabsContent value="approved" className="mt-0">
          <AgentTable
            agents={approvedAgents}
            onViewDetails={handleViewDetails}
            onApprove={handleApprove}
            onReject={handleRejectClick}
            onBlock={handleBlockClick}
            onUnblock={handleUnblock}
            onRestore={handleRestore}
          />
        </TabsContent>

        <TabsContent value="rejected" className="mt-0">
          <AgentTable
            agents={rejectedAgents}
            onViewDetails={handleViewDetails}
            onApprove={handleApprove}
            onReject={handleRejectClick}
            onBlock={handleBlockClick}
            onUnblock={handleUnblock}
            onRestore={handleRestore}
          />
        </TabsContent>

        <TabsContent value="blocked" className="mt-0">
          <AgentTable
            agents={blockedAgents}
            onViewDetails={handleViewDetails}
            onApprove={handleApprove}
            onReject={handleRejectClick}
            onBlock={handleBlockClick}
            onUnblock={handleUnblock}
            onRestore={handleRestore}
          />
        </TabsContent>
      </Tabs>

      <AgentDetailsDialog agent={selectedAgent} open={isDetailsOpen} onOpenChange={setIsDetailsOpen} />

      <BlockAgentDialog
        agent={agentToBlock}
        open={isBlockDialogOpen}
        onOpenChange={setIsBlockDialogOpen}
        onConfirm={handleBlockConfirm}
      />

      <RejectAgentDialog
        agent={agentToReject}
        open={isRejectDialogOpen}
        onOpenChange={setIsRejectDialogOpen}
        onConfirm={handleRejectConfirm}
      />
    </section>
  )
}
