"use client"

import { useState, useMemo, useCallback } from "react"
import { Tabs, TabsContent, TabsList, TabsTrigger } from "@/components/ui/tabs"
import { Button } from "@/components/ui/button"
import { Plus } from "lucide-react"
import { CityTable } from "./city-table"
import { AddCityDialog } from "./add-city-dialog"
import { EditCityDialog } from "./edit-city-dialog"
import { DeleteCityDialog } from "./delete-city-dialog"
import { locationDataStore, type City, type State } from "@/lib/data/locations"
import { useToastEnhanced } from "@/hooks/use-toast-enhanced"

export function CitiesClient() {
  const toast = useToastEnhanced()
  const [currentTab, setCurrentTab] = useState("all")

  const [cities, setCities] = useState<City[]>(locationDataStore.getAllCities())
  const [states] = useState<State[]>(locationDataStore.getActiveStates())
  const [isAddDialogOpen, setIsAddDialogOpen] = useState(false)
  const [editingCity, setEditingCity] = useState<City | null>(null)
  const [deletingCity, setDeletingCity] = useState<City | null>(null)

  const activeCities = useMemo(() => cities.filter((c) => c.status === "active"), [cities])
  const inactiveCities = useMemo(() => cities.filter((c) => c.status === "inactive"), [cities])

  const handleTabChange = (value: string) => {
    setCurrentTab(value)
  }

  const handleAddCity = useCallback(
    (stateId: string, name: string) => {
      const newCity = locationDataStore.addCity(stateId, name)
      if (newCity) {
        setCities(locationDataStore.getAllCities())
        toast.success({
          title: "City Added",
          description: `${name} has been added successfully.`,
        })
        setIsAddDialogOpen(false)
        return true
      }
      toast.error({
        title: "Failed to Add City",
        description: "A city with this name already exists in the selected state.",
      })
      return false
    },
    [toast],
  )

  const handleEditCity = useCallback(
    (id: string, stateId: string, name: string) => {
      const updated = locationDataStore.updateCity(id, stateId, name)
      if (updated) {
        setCities(locationDataStore.getAllCities())
        toast.success({
          title: "City Updated",
          description: `${name} has been updated successfully.`,
        })
        setEditingCity(null)
        return true
      }
      toast.error({
        title: "Failed to Update City",
        description: "A city with this name already exists in the selected state.",
      })
      return false
    },
    [toast],
  )

  const handleToggleStatus = useCallback(
    (city: City) => {
      const updated = locationDataStore.toggleCityStatus(city.id)
      if (updated) {
        setCities(locationDataStore.getAllCities())
        const action = updated.status === "active" ? "activated" : "deactivated"
        toast.success({
          title: `City ${action.charAt(0).toUpperCase() + action.slice(1)}`,
          description: `${city.name} has been ${action}.`,
        })
      }
    },
    [toast],
  )

  const handleDeleteCity = useCallback(() => {
    if (!deletingCity) return

    const deleted = locationDataStore.deleteCity(deletingCity.id)
    if (deleted) {
      setCities(locationDataStore.getAllCities())
      toast.success({
        title: "City Deleted",
        description: `${deletingCity.name} has been deleted.`,
      })
    }
    setDeletingCity(null)
  }, [deletingCity, toast])

  const getDisplayCities = () => {
    switch (currentTab) {
      case "active":
        return activeCities
      case "inactive":
        return inactiveCities
      default:
        return cities
    }
  }

  return (
    <section aria-labelledby="cities-heading" className="admin-section">
      <header className="admin-page-header flex flex-col gap-4 sm:flex-row sm:items-center sm:justify-between">
        <div>
          <h1 id="cities-heading" className="admin-page-title">
            Cities
          </h1>
          <p className="admin-page-description">Manage cities for your platform coverage.</p>
        </div>
        <Button onClick={() => setIsAddDialogOpen(true)} className="w-full sm:w-auto">
          <Plus className="mr-2 h-4 w-4" aria-hidden="true" />
          Add City
        </Button>
      </header>

      <Tabs value={currentTab} onValueChange={handleTabChange} className="w-full">
        <TabsList className="mb-6 h-auto p-1.5 bg-muted/50 rounded-xl border border-border/50">
          <TabsTrigger
            value="all"
            className="rounded-lg data-[state=active]:bg-background data-[state=active]:shadow-sm px-4 py-2"
          >
            All ({cities.length})
          </TabsTrigger>
          <TabsTrigger
            value="active"
            className="rounded-lg data-[state=active]:bg-background data-[state=active]:shadow-sm px-4 py-2"
          >
            Active ({activeCities.length})
          </TabsTrigger>
          <TabsTrigger
            value="inactive"
            className="rounded-lg data-[state=active]:bg-background data-[state=active]:shadow-sm px-4 py-2"
          >
            Inactive ({inactiveCities.length})
          </TabsTrigger>
        </TabsList>

        <TabsContent value="all" className="mt-0">
          <CityTable
            cities={getDisplayCities()}
            onEdit={setEditingCity}
            onDelete={setDeletingCity}
            onToggleStatus={handleToggleStatus}
          />
        </TabsContent>

        <TabsContent value="active" className="mt-0">
          <CityTable
            cities={getDisplayCities()}
            onEdit={setEditingCity}
            onDelete={setDeletingCity}
            onToggleStatus={handleToggleStatus}
          />
        </TabsContent>

        <TabsContent value="inactive" className="mt-0">
          <CityTable
            cities={getDisplayCities()}
            onEdit={setEditingCity}
            onDelete={setDeletingCity}
            onToggleStatus={handleToggleStatus}
          />
        </TabsContent>
      </Tabs>

      <AddCityDialog states={states} open={isAddDialogOpen} onOpenChange={setIsAddDialogOpen} onAdd={handleAddCity} />

      <EditCityDialog
        city={editingCity}
        states={states}
        open={!!editingCity}
        onOpenChange={(open) => !open && setEditingCity(null)}
        onSave={handleEditCity}
      />

      <DeleteCityDialog
        city={deletingCity}
        open={!!deletingCity}
        onOpenChange={(open) => !open && setDeletingCity(null)}
        onConfirm={handleDeleteCity}
      />
    </section>
  )
}
