"use client"

import { Dialog, DialogContent, DialogHeader, DialogTitle, DialogDescription } from "@/components/ui/dialog"
import { Badge } from "@/components/ui/badge"
import { Separator } from "@/components/ui/separator"
import { Table, TableBody, TableCell, TableHead, TableHeader, TableRow } from "@/components/ui/table"
import { User, Mail, Phone, Calendar, IndianRupee, ShoppingBag } from "lucide-react"
import type { Customer } from "@/lib/data/customers"

interface CustomerDetailsDialogProps {
  customer: Customer | null
  open: boolean
  onOpenChange: (open: boolean) => void
}

export function CustomerDetailsDialog({ customer, open, onOpenChange }: CustomerDetailsDialogProps) {
  if (!customer) return null

  const formatCurrency = (amount: number) => {
    return new Intl.NumberFormat("en-IN", {
      style: "currency",
      currency: "INR",
      maximumFractionDigits: 0,
    }).format(amount)
  }

  const formatDate = (dateString: string) => {
    return new Date(dateString).toLocaleDateString("en-IN", {
      year: "numeric",
      month: "long",
      day: "numeric",
    })
  }

  return (
    <Dialog open={open} onOpenChange={onOpenChange}>
      <DialogContent className="max-h-[90vh] max-w-2xl overflow-y-auto">
        <DialogHeader>
          <DialogTitle className="flex items-center gap-2">
            <User className="h-5 w-5" aria-hidden="true" />
            {customer.firstName} {customer.lastName}
          </DialogTitle>
          <DialogDescription>Customer details and booking history</DialogDescription>
        </DialogHeader>

        <div className="space-y-6">
          {/* Customer Info */}
          <section aria-labelledby="customer-info-heading">
            <h3 id="customer-info-heading" className="sr-only">
              Customer Information
            </h3>
            <div className="grid gap-4 sm:grid-cols-2">
              <div className="flex items-center gap-2 text-sm">
                <Mail className="h-4 w-4 text-muted-foreground" aria-hidden="true" />
                <span>{customer.email}</span>
              </div>
              <div className="flex items-center gap-2 text-sm">
                <Phone className="h-4 w-4 text-muted-foreground" aria-hidden="true" />
                <span>{customer.phone}</span>
              </div>
              <div className="flex items-center gap-2 text-sm">
                <Calendar className="h-4 w-4 text-muted-foreground" aria-hidden="true" />
                <span>Joined: {formatDate(customer.joinedAt)}</span>
              </div>
              <div className="flex items-center gap-2 text-sm">
                <Badge variant={customer.status === "active" ? "default" : "destructive"}>
                  {customer.status === "active" ? "Active" : "Blocked"}
                </Badge>
              </div>
            </div>

            {customer.status === "blocked" && customer.blockReason && (
              <div className="mt-4 rounded-lg bg-destructive/10 p-4">
                <p className="text-sm font-medium text-destructive">Block Reason:</p>
                <p className="mt-1 text-sm text-muted-foreground">{customer.blockReason}</p>
                {customer.blockedAt && (
                  <p className="mt-2 text-xs text-muted-foreground">Blocked on: {formatDate(customer.blockedAt)}</p>
                )}
              </div>
            )}
          </section>

          <Separator />

          {/* Statistics */}
          <section aria-labelledby="statistics-heading">
            <h3 id="statistics-heading" className="mb-4 text-lg font-semibold">
              Statistics
            </h3>
            <div className="grid gap-4 sm:grid-cols-3">
              <div className="rounded-lg border p-4">
                <div className="flex items-center gap-2">
                  <ShoppingBag className="h-4 w-4 text-muted-foreground" aria-hidden="true" />
                  <span className="text-sm text-muted-foreground">Total Bookings</span>
                </div>
                <p className="mt-2 text-2xl font-bold">{customer.totalBookings}</p>
              </div>
              <div className="rounded-lg border p-4">
                <div className="flex items-center gap-2">
                  <IndianRupee className="h-4 w-4 text-muted-foreground" aria-hidden="true" />
                  <span className="text-sm text-muted-foreground">Total Spent</span>
                </div>
                <p className="mt-2 text-2xl font-bold">{formatCurrency(customer.totalSpent)}</p>
              </div>
              <div className="rounded-lg border p-4">
                <div className="flex items-center gap-2">
                  <Calendar className="h-4 w-4 text-muted-foreground" aria-hidden="true" />
                  <span className="text-sm text-muted-foreground">Last Active</span>
                </div>
                <p className="mt-2 text-sm font-medium">{formatDate(customer.lastActive)}</p>
              </div>
            </div>
          </section>

          <Separator />

          {/* Booking History */}
          <section aria-labelledby="bookings-heading">
            <h3 id="bookings-heading" className="mb-4 text-lg font-semibold">
              Booking History
            </h3>
            {customer.bookings.length === 0 ? (
              <p className="text-center text-sm text-muted-foreground py-4">No bookings found for this customer.</p>
            ) : (
              <div className="rounded-lg border">
                <Table>
                  <TableHeader>
                    <TableRow>
                      <TableHead>Package</TableHead>
                      <TableHead>Destination</TableHead>
                      <TableHead>Date</TableHead>
                      <TableHead>Amount</TableHead>
                      <TableHead>Status</TableHead>
                    </TableRow>
                  </TableHeader>
                  <TableBody>
                    {customer.bookings.map((booking) => (
                      <TableRow key={booking.id}>
                        <TableCell className="font-medium">{booking.packageName}</TableCell>
                        <TableCell>{booking.destination}</TableCell>
                        <TableCell>{formatDate(booking.travelDate)}</TableCell>
                        <TableCell>{formatCurrency(booking.amount)}</TableCell>
                        <TableCell>
                          <Badge
                            variant={
                              booking.status === "completed"
                                ? "default"
                                : booking.status === "upcoming"
                                  ? "secondary"
                                  : "destructive"
                            }
                          >
                            {booking.status.charAt(0).toUpperCase() + booking.status.slice(1)}
                          </Badge>
                        </TableCell>
                      </TableRow>
                    ))}
                  </TableBody>
                </Table>
              </div>
            )}
          </section>
        </div>
      </DialogContent>
    </Dialog>
  )
}
