"use client"

import { useState, useMemo, useCallback } from "react"
import { useRouter } from "next/navigation"
import { Tabs, TabsContent, TabsList, TabsTrigger } from "@/components/ui/tabs"
import { Button } from "@/components/ui/button"
import { Plus } from "lucide-react"
import { FixPackagesTable } from "./fix-packages-table"
import { DeleteFixPackageDialog } from "./delete-fix-package-dialog"
import { ViewPackageDetailsDialog } from "./view-package-details-dialog"
import { fixPackageDataStore, type FixPackage } from "@/lib/data/fix-packages"
import { useToastEnhanced } from "@/hooks/use-toast-enhanced"

export function FixPackagesClient() {
  const toast = useToastEnhanced()
  const router = useRouter()
  const [currentTab, setCurrentTab] = useState("active")

  const [packages, setPackages] = useState<FixPackage[]>(fixPackageDataStore.getAllPackages())
  const [deletingPackage, setDeletingPackage] = useState<FixPackage | null>(null)
  const [viewingPackage, setViewingPackage] = useState<FixPackage | null>(null)

  const activePackages = useMemo(() => packages.filter((p) => p.status === "active"), [packages])
  const inactivePackages = useMemo(() => packages.filter((p) => p.status === "inactive"), [packages])

  const handleTabChange = (value: string) => {
    setCurrentTab(value)
  }

  const handleToggleStatus = useCallback(
    (pkg: FixPackage) => {
      const updated = fixPackageDataStore.togglePackageStatus(pkg.id)
      if (updated) {
        setPackages(fixPackageDataStore.getAllPackages())
        const action = updated.status === "active" ? "activated" : "deactivated"
        toast.success({
          title: `Package ${action.charAt(0).toUpperCase() + action.slice(1)}`,
          description: `${pkg.name} has been ${action}.`,
        })
      }
    },
    [toast],
  )

  const handleDeletePackage = useCallback(() => {
    if (!deletingPackage) return

    const deleted = fixPackageDataStore.deletePackage(deletingPackage.id)
    if (deleted) {
      setPackages(fixPackageDataStore.getAllPackages())
      toast.success({
        title: "Package Deleted",
        description: `${deletingPackage.name} has been deleted.`,
      })
    }
    setDeletingPackage(null)
  }, [deletingPackage, toast])

  const handleEdit = (pkg: FixPackage) => {
    router.push(`/admin/fix-packages/edit/${pkg.id}`)
  }

  const getDisplayPackages = () => {
    switch (currentTab) {
      case "active":
        return activePackages
      case "inactive":
        return inactivePackages
      default:
        return packages
    }
  }

  return (
    <section aria-labelledby="fix-packages-heading" className="admin-section">
      <header className="admin-page-header flex flex-col gap-4 sm:flex-row sm:items-center sm:justify-between">
        <div>
          <h1 id="fix-packages-heading" className="admin-page-title">
            Fix Packages
          </h1>
          <p className="admin-page-description">Manage all fixed tour packages with pricing and itinerary details.</p>
        </div>
        <Button onClick={() => router.push("/admin/fix-packages/add-new")} className="w-full sm:w-auto">
          <Plus className="mr-2 h-4 w-4" aria-hidden="true" />
          Add New Package
        </Button>
      </header>

      <Tabs value={currentTab} onValueChange={handleTabChange} className="w-full">
        <TabsList className="mb-6 h-auto p-1.5 bg-muted/50 rounded-xl border border-border/50">
          <TabsTrigger
            value="active"
            className="rounded-lg data-[state=active]:bg-background data-[state=active]:shadow-sm px-4 py-2"
          >
            Active ({activePackages.length})
          </TabsTrigger>
          <TabsTrigger
            value="inactive"
            className="rounded-lg data-[state=active]:bg-background data-[state=active]:shadow-sm px-4 py-2"
          >
            Inactive ({inactivePackages.length})
          </TabsTrigger>
        </TabsList>

        <TabsContent value="active" className="mt-0">
          <FixPackagesTable
            packages={getDisplayPackages()}
            onView={setViewingPackage}
            onEdit={handleEdit}
            onDelete={setDeletingPackage}
            onToggleStatus={handleToggleStatus}
          />
        </TabsContent>

        <TabsContent value="inactive" className="mt-0">
          <FixPackagesTable
            packages={getDisplayPackages()}
            onView={setViewingPackage}
            onEdit={handleEdit}
            onDelete={setDeletingPackage}
            onToggleStatus={handleToggleStatus}
          />
        </TabsContent>
      </Tabs>

      <DeleteFixPackageDialog
        package={deletingPackage}
        open={!!deletingPackage}
        onOpenChange={(open) => !open && setDeletingPackage(null)}
        onConfirm={handleDeletePackage}
      />

      <ViewPackageDetailsDialog
        package={viewingPackage}
        open={!!viewingPackage}
        onOpenChange={(open) => !open && setViewingPackage(null)}
      />
    </section>
  )
}
