"use client"

import { useState, useMemo } from "react"
import { Table, TableBody, TableCell, TableHead, TableHeader, TableRow } from "@/components/ui/table"
import { Button } from "@/components/ui/button"
import { Badge } from "@/components/ui/badge"
import { Input } from "@/components/ui/input"
import {
  Pencil,
  Trash2,
  Search,
  FileText,
  ChevronLeft,
  ChevronRight,
  ArrowUpDown,
  ArrowUp,
  ArrowDown,
} from "lucide-react"
import type { HotelPartner } from "@/lib/data/hotel-partners"
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from "@/components/ui/select"

interface HotelPartnerTableProps {
  partners: HotelPartner[]
  onViewDocuments: (partner: HotelPartner) => void
  onEdit: (partner: HotelPartner) => void
  onDelete: (partner: HotelPartner) => void
  onToggleStatus: (partner: HotelPartner) => void
}

type SortField = "hotelName" | "ownerName" | "stateName" | "cityName" | "email"
type SortDirection = "asc" | "desc"

const ITEMS_PER_PAGE_OPTIONS = [10, 20, 50, 100]

export function HotelPartnerTable({
  partners,
  onViewDocuments,
  onEdit,
  onDelete,
  onToggleStatus,
}: HotelPartnerTableProps) {
  const [searchQuery, setSearchQuery] = useState("")
  const [currentPage, setCurrentPage] = useState(1)
  const [itemsPerPage, setItemsPerPage] = useState(10)
  const [sortField, setSortField] = useState<SortField>("hotelName")
  const [sortDirection, setSortDirection] = useState<SortDirection>("asc")

  // Filter data based on search
  const filteredData = useMemo(() => {
    if (!searchQuery.trim()) return partners
    const query = searchQuery.toLowerCase()
    return partners.filter(
      (p) =>
        p.hotelName.toLowerCase().includes(query) ||
        p.ownerName.toLowerCase().includes(query) ||
        p.email.toLowerCase().includes(query) ||
        p.phone.includes(query) ||
        p.stateName.toLowerCase().includes(query) ||
        p.cityName.toLowerCase().includes(query),
    )
  }, [partners, searchQuery])

  // Sort data
  const sortedData = useMemo(() => {
    return [...filteredData].sort((a, b) => {
      let comparison = 0
      switch (sortField) {
        case "hotelName":
          comparison = a.hotelName.localeCompare(b.hotelName)
          break
        case "ownerName":
          comparison = a.ownerName.localeCompare(b.ownerName)
          break
        case "stateName":
          comparison = a.stateName.localeCompare(b.stateName)
          break
        case "cityName":
          comparison = a.cityName.localeCompare(b.cityName)
          break
        case "email":
          comparison = a.email.localeCompare(b.email)
          break
      }
      return sortDirection === "asc" ? comparison : -comparison
    })
  }, [filteredData, sortField, sortDirection])

  // Paginate data
  const paginatedData = useMemo(() => {
    const startIndex = (currentPage - 1) * itemsPerPage
    return sortedData.slice(startIndex, startIndex + itemsPerPage)
  }, [sortedData, currentPage, itemsPerPage])

  const totalPages = Math.ceil(sortedData.length / itemsPerPage)

  const handleSort = (field: SortField) => {
    if (sortField === field) {
      setSortDirection(sortDirection === "asc" ? "desc" : "asc")
    } else {
      setSortField(field)
      setSortDirection("asc")
    }
  }

  const getSortIcon = (field: SortField) => {
    if (sortField !== field) {
      return <ArrowUpDown className="ml-1 h-3 w-3 text-muted-foreground" />
    }
    return sortDirection === "asc" ? <ArrowUp className="ml-1 h-3 w-3" /> : <ArrowDown className="ml-1 h-3 w-3" />
  }

  const handlePageChange = (page: number) => {
    setCurrentPage(Math.max(1, Math.min(page, totalPages)))
  }

  const handleItemsPerPageChange = (value: string) => {
    setItemsPerPage(Number(value))
    setCurrentPage(1)
  }

  return (
    <div className="space-y-4">
      {/* Search and filters */}
      <div className="flex flex-col gap-4 sm:flex-row sm:items-center sm:justify-between">
        <div className="relative max-w-sm">
          <Search
            className="absolute left-3 top-1/2 h-4 w-4 -translate-y-1/2 text-muted-foreground"
            aria-hidden="true"
          />
          <Input
            type="search"
            placeholder="Search by name, email, phone, location..."
            value={searchQuery}
            onChange={(e) => {
              setSearchQuery(e.target.value)
              setCurrentPage(1)
            }}
            className="pl-9"
            aria-label="Search hotel partners"
          />
        </div>
        <div className="flex items-center gap-2">
          <span className="text-sm text-muted-foreground">Show:</span>
          <Select value={String(itemsPerPage)} onValueChange={handleItemsPerPageChange}>
            <SelectTrigger className="w-20" aria-label="Items per page">
              <SelectValue />
            </SelectTrigger>
            <SelectContent>
              {ITEMS_PER_PAGE_OPTIONS.map((option) => (
                <SelectItem key={option} value={String(option)}>
                  {option}
                </SelectItem>
              ))}
            </SelectContent>
          </Select>
        </div>
      </div>

      {/* Results count */}
      <div className="flex items-center justify-between text-sm text-muted-foreground">
        <p>
          Showing {paginatedData.length > 0 ? (currentPage - 1) * itemsPerPage + 1 : 0} to{" "}
          {Math.min(currentPage * itemsPerPage, sortedData.length)} of {sortedData.length} partners
        </p>
        {searchQuery && (
          <Badge variant="secondary" className="font-normal">
            Filtered from {partners.length} total
          </Badge>
        )}
      </div>

      {/* Table */}
      <div className="rounded-md border overflow-x-auto">
        <Table>
          <TableHeader>
            <TableRow>
              <TableHead className="w-16 text-center">S.No</TableHead>
              <TableHead>
                <Button
                  variant="ghost"
                  size="sm"
                  className="-ml-3 h-8 data-[state=open]:bg-accent"
                  onClick={() => handleSort("hotelName")}
                >
                  Hotel Name
                  {getSortIcon("hotelName")}
                </Button>
              </TableHead>
              <TableHead className="hidden md:table-cell">
                <Button
                  variant="ghost"
                  size="sm"
                  className="-ml-3 h-8 data-[state=open]:bg-accent"
                  onClick={() => handleSort("ownerName")}
                >
                  Owner Name
                  {getSortIcon("ownerName")}
                </Button>
              </TableHead>
              <TableHead className="hidden lg:table-cell">
                <Button
                  variant="ghost"
                  size="sm"
                  className="-ml-3 h-8 data-[state=open]:bg-accent"
                  onClick={() => handleSort("stateName")}
                >
                  State
                  {getSortIcon("stateName")}
                </Button>
              </TableHead>
              <TableHead className="hidden lg:table-cell">
                <Button
                  variant="ghost"
                  size="sm"
                  className="-ml-3 h-8 data-[state=open]:bg-accent"
                  onClick={() => handleSort("cityName")}
                >
                  City
                  {getSortIcon("cityName")}
                </Button>
              </TableHead>
              <TableHead className="hidden xl:table-cell">
                <Button
                  variant="ghost"
                  size="sm"
                  className="-ml-3 h-8 data-[state=open]:bg-accent"
                  onClick={() => handleSort("email")}
                >
                  Email
                  {getSortIcon("email")}
                </Button>
              </TableHead>
              <TableHead className="hidden md:table-cell">Phone</TableHead>
              <TableHead>Documents</TableHead>
              <TableHead>Status</TableHead>
              <TableHead className="text-right">Actions</TableHead>
            </TableRow>
          </TableHeader>
          <TableBody>
            {paginatedData.length === 0 ? (
              <TableRow>
                <TableCell colSpan={10} className="h-24 text-center">
                  <p className="text-muted-foreground">
                    {searchQuery ? `No partners found matching "${searchQuery}"` : "No hotel partners found."}
                  </p>
                </TableCell>
              </TableRow>
            ) : (
              paginatedData.map((partner, index) => (
                <TableRow key={partner.id}>
                  <TableCell className="text-center font-medium text-muted-foreground">
                    {(currentPage - 1) * itemsPerPage + index + 1}
                  </TableCell>
                  <TableCell className="font-medium">{partner.hotelName}</TableCell>
                  <TableCell className="hidden md:table-cell text-muted-foreground">{partner.ownerName}</TableCell>
                  <TableCell className="hidden lg:table-cell text-muted-foreground">{partner.stateName}</TableCell>
                  <TableCell className="hidden lg:table-cell text-muted-foreground">{partner.cityName}</TableCell>
                  <TableCell className="hidden xl:table-cell text-muted-foreground">{partner.email}</TableCell>
                  <TableCell className="hidden md:table-cell text-muted-foreground">{partner.phone}</TableCell>
                  <TableCell>
                    <Button
                      variant="outline"
                      size="sm"
                      onClick={() => onViewDocuments(partner)}
                      aria-label={`View documents for ${partner.hotelName}`}
                    >
                      <FileText className="mr-1 h-4 w-4" aria-hidden="true" />
                      View
                    </Button>
                  </TableCell>
                  <TableCell>
                    <Button
                      variant="ghost"
                      size="sm"
                      onClick={() => onToggleStatus(partner)}
                      className="p-0 h-auto"
                      aria-label={`Toggle status for ${partner.hotelName}`}
                    >
                      <Badge variant={partner.status === "active" ? "default" : "secondary"}>
                        {partner.status === "active" ? "Active" : "Inactive"}
                      </Badge>
                    </Button>
                  </TableCell>
                  <TableCell className="text-right">
                    <div className="flex items-center justify-end gap-1">
                      <Button
                        variant="outline"
                        size="icon"
                        onClick={() => onEdit(partner)}
                        aria-label={`Edit ${partner.hotelName}`}
                      >
                        <Pencil className="h-4 w-4" aria-hidden="true" />
                      </Button>
                      <Button
                        variant="destructive"
                        size="icon"
                        onClick={() => onDelete(partner)}
                        aria-label={`Delete ${partner.hotelName}`}
                      >
                        <Trash2 className="h-4 w-4" aria-hidden="true" />
                      </Button>
                    </div>
                  </TableCell>
                </TableRow>
              ))
            )}
          </TableBody>
        </Table>
      </div>

      {/* Pagination */}
      {totalPages > 1 && (
        <div className="flex items-center justify-between">
          <p className="text-sm text-muted-foreground">
            Page {currentPage} of {totalPages}
          </p>
          <div className="flex items-center gap-1">
            <Button
              variant="outline"
              size="icon"
              onClick={() => handlePageChange(currentPage - 1)}
              disabled={currentPage === 1}
              aria-label="Previous page"
            >
              <ChevronLeft className="h-4 w-4" />
            </Button>
            <div className="hidden sm:flex items-center gap-1">
              {Array.from({ length: Math.min(5, totalPages) }, (_, i) => {
                let pageNum: number
                if (totalPages <= 5) {
                  pageNum = i + 1
                } else if (currentPage <= 3) {
                  pageNum = i + 1
                } else if (currentPage >= totalPages - 2) {
                  pageNum = totalPages - 4 + i
                } else {
                  pageNum = currentPage - 2 + i
                }
                return (
                  <Button
                    key={pageNum}
                    variant={currentPage === pageNum ? "default" : "outline"}
                    size="icon"
                    onClick={() => handlePageChange(pageNum)}
                    aria-label={`Go to page ${pageNum}`}
                    aria-current={currentPage === pageNum ? "page" : undefined}
                  >
                    {pageNum}
                  </Button>
                )
              })}
            </div>
            <Button
              variant="outline"
              size="icon"
              onClick={() => handlePageChange(currentPage + 1)}
              disabled={currentPage === totalPages}
              aria-label="Next page"
            >
              <ChevronRight className="h-4 w-4" />
            </Button>
          </div>
        </div>
      )}
    </div>
  )
}
