"use client"

import { useState, useEffect } from "react"
import { useForm } from "react-hook-form"
import { zodResolver } from "@hookform/resolvers/zod"
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from "@/components/ui/card"
import { Button } from "@/components/ui/button"
import { Input } from "@/components/ui/input"
import { Textarea } from "@/components/ui/textarea"
import { Label } from "@/components/ui/label"
import { Loader2, Save, AlertCircle, CheckCircle2 } from "lucide-react"
import { RichTextEditor } from "@/components/shared/rich-text-editor"
import { legalPageSchema, type LegalPageInput } from "@/lib/validations/legal"
import { getLegalPage, updateLegalPage, type LegalPageSlug, type LegalPage } from "@/lib/data/legal-pages"
import { useToastEnhanced } from "@/hooks/use-toast-enhanced"
import { Alert, AlertDescription } from "@/components/ui/alert"

interface LegalPageEditorProps {
  slug: LegalPageSlug
  title: string
  description: string
}

export function LegalPageEditor({ slug, title, description }: LegalPageEditorProps) {
  const [isLoading, setIsLoading] = useState(false)
  const [pageData, setPageData] = useState<LegalPage | undefined>()
  const [content, setContent] = useState("")
  const [lastSaved, setLastSaved] = useState<Date | null>(null)
  const toast = useToastEnhanced()

  const {
    register,
    handleSubmit,
    setValue,
    formState: { errors, isDirty },
    reset,
  } = useForm<LegalPageInput>({
    resolver: zodResolver(legalPageSchema),
    defaultValues: {
      metaTitle: "",
      metaDescription: "",
      content: "",
    },
  })

  // Load initial data
  useEffect(() => {
    const data = getLegalPage(slug)
    if (data) {
      setPageData(data)
      setContent(data.content)
      reset({
        metaTitle: data.metaTitle,
        metaDescription: data.metaDescription,
        content: data.content,
      })
      setLastSaved(data.updatedAt)
    }
  }, [slug, reset])

  // Update form when content changes
  useEffect(() => {
    setValue("content", content, { shouldDirty: true })
  }, [content, setValue])

  const onSubmit = async (data: LegalPageInput) => {
    setIsLoading(true)

    try {
      // Simulate network delay for realistic UX
      await new Promise((resolve) => setTimeout(resolve, 500))

      const updated = updateLegalPage(slug, {
        metaTitle: data.metaTitle,
        metaDescription: data.metaDescription,
        content: data.content,
      })

      setPageData(updated)
      setLastSaved(updated.updatedAt)
      reset(data)

      toast.success({
        title: "Changes saved",
        description: `${title} has been updated successfully.`,
      })
    } catch (error) {
      toast.error({
        title: "Failed to save",
        description: "There was an error saving your changes. Please try again.",
      })
    } finally {
      setIsLoading(false)
    }
  }

  const metaTitleLength = (register("metaTitle").name && pageData?.metaTitle.length) || 0
  const metaDescLength = (register("metaDescription").name && pageData?.metaDescription.length) || 0

  return (
    <section className="space-y-6">
      <div className="flex flex-col gap-4 sm:flex-row sm:items-center sm:justify-between">
        <div className="space-y-1">
          <h1 className="text-2xl font-bold tracking-tight">{title}</h1>
          <p className="text-sm text-muted-foreground">{description}</p>
        </div>
        {lastSaved && (
          <div className="flex items-center gap-2 text-sm text-muted-foreground">
            <CheckCircle2 className="h-4 w-4 text-green-500" />
            <span>
              Last saved:{" "}
              {lastSaved.toLocaleDateString("en-IN", {
                day: "numeric",
                month: "short",
                year: "numeric",
                hour: "2-digit",
                minute: "2-digit",
              })}
            </span>
          </div>
        )}
      </div>

      <form onSubmit={handleSubmit(onSubmit)} className="space-y-6">
        {/* SEO Settings Card */}
        <Card>
          <CardHeader>
            <CardTitle className="text-lg">SEO Settings</CardTitle>
            <CardDescription>
              Optimize your page for search engines. Keep meta title under 70 characters and description under 160.
            </CardDescription>
          </CardHeader>
          <CardContent className="space-y-4">
            <div className="space-y-2">
              <div className="flex items-center justify-between">
                <Label htmlFor="metaTitle">Meta Title</Label>
                <span className="text-xs text-muted-foreground">{pageData?.metaTitle.length || 0}/70</span>
              </div>
              <Input
                id="metaTitle"
                placeholder="Enter meta title for SEO"
                {...register("metaTitle")}
                aria-describedby={errors.metaTitle ? "metaTitle-error" : undefined}
                className={errors.metaTitle ? "border-destructive" : ""}
              />
              {errors.metaTitle && (
                <p id="metaTitle-error" className="text-sm text-destructive flex items-center gap-1">
                  <AlertCircle className="h-3 w-3" />
                  {errors.metaTitle.message}
                </p>
              )}
            </div>

            <div className="space-y-2">
              <div className="flex items-center justify-between">
                <Label htmlFor="metaDescription">Meta Description</Label>
                <span className="text-xs text-muted-foreground">{pageData?.metaDescription.length || 0}/160</span>
              </div>
              <Textarea
                id="metaDescription"
                placeholder="Enter meta description for SEO (150-160 characters recommended)"
                rows={3}
                {...register("metaDescription")}
                aria-describedby={errors.metaDescription ? "metaDescription-error" : undefined}
                className={errors.metaDescription ? "border-destructive" : ""}
              />
              {errors.metaDescription && (
                <p id="metaDescription-error" className="text-sm text-destructive flex items-center gap-1">
                  <AlertCircle className="h-3 w-3" />
                  {errors.metaDescription.message}
                </p>
              )}
            </div>
          </CardContent>
        </Card>

        {/* Content Editor Card */}
        <Card>
          <CardHeader>
            <CardTitle className="text-lg">Page Content</CardTitle>
            <CardDescription>Write and format your legal content using the rich text editor below.</CardDescription>
          </CardHeader>
          <CardContent className="space-y-4">
            <RichTextEditor content={content} onChange={setContent} placeholder="Start writing your content here..." />
            {errors.content && (
              <Alert variant="destructive">
                <AlertCircle className="h-4 w-4" />
                <AlertDescription>{errors.content.message}</AlertDescription>
              </Alert>
            )}
          </CardContent>
        </Card>

        {/* Save Button */}
        <div className="flex items-center justify-end gap-4">
          {isDirty && <p className="text-sm text-amber-600 dark:text-amber-400">You have unsaved changes</p>}
          <Button type="submit" disabled={isLoading || !isDirty} size="lg">
            {isLoading ? (
              <>
                <Loader2 className="mr-2 h-4 w-4 animate-spin" />
                Saving...
              </>
            ) : (
              <>
                <Save className="mr-2 h-4 w-4" />
                Save Changes
              </>
            )}
          </Button>
        </div>
      </form>
    </section>
  )
}
