"use client"

import { Button } from "@/components/ui/button"
import { Badge } from "@/components/ui/badge"
import { Card } from "@/components/ui/card"
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from "@/components/ui/select"
import { Eye, Edit, Trash2, ToggleLeft, ToggleRight } from "lucide-react"
import { useState } from "react"
import type { SpiritualPackage } from "@/lib/data/spiritual-packages"
import { ViewPackageDetailsDialog } from "./view-package-details-dialog"

interface SpiritualPackagesTableProps {
  packages: SpiritualPackage[]
  onToggleStatus: (id: string) => void
  onEdit: (id: string) => void
  onDelete: (id: string, name: string) => void
  currentPage: number
  totalPages: number
  itemsPerPage: number
  onPageChange: (page: number) => void
  onItemsPerPageChange: (items: number) => void
}

export function SpiritualPackagesTable({
  packages,
  onToggleStatus,
  onEdit,
  onDelete,
  currentPage,
  totalPages,
  itemsPerPage,
  onPageChange,
  onItemsPerPageChange,
}: SpiritualPackagesTableProps) {
  const [viewDialogOpen, setViewDialogOpen] = useState(false)
  const [selectedPackage, setSelectedPackage] = useState<SpiritualPackage | null>(null)

  const handleViewDetails = (pkg: SpiritualPackage) => {
    setSelectedPackage(pkg)
    setViewDialogOpen(true)
  }

  return (
    <>
      <Card>
        <div className="overflow-x-auto">
          <table className="w-full">
            <thead>
              <tr className="border-b bg-muted/50">
                <th className="p-4 text-left text-sm font-medium">S.No</th>
                <th className="p-4 text-left text-sm font-medium">City Name</th>
                <th className="p-4 text-left text-sm font-medium">Details</th>
                <th className="p-4 text-left text-sm font-medium">Total Days</th>
                <th className="p-4 text-left text-sm font-medium">Status</th>
                <th className="p-4 text-right text-sm font-medium">Actions</th>
              </tr>
            </thead>
            <tbody>
              {packages.length === 0 ? (
                <tr>
                  <td colSpan={6} className="p-8 text-center text-muted-foreground">
                    No packages found
                  </td>
                </tr>
              ) : (
                packages.map((pkg, index) => (
                  <tr key={pkg.id} className="border-b last:border-0 hover:bg-muted/30">
                    <td className="p-4 text-sm">{(currentPage - 1) * itemsPerPage + index + 1}</td>
                    <td className="p-4">
                      <div className="font-medium">{pkg.cityName}</div>
                      <div className="text-xs text-muted-foreground">{pkg.totalLocations} locations</div>
                    </td>
                    <td className="p-4">
                      <Button variant="outline" size="sm" onClick={() => handleViewDetails(pkg)}>
                        <Eye className="mr-2 h-4 w-4" />
                        View Details
                      </Button>
                    </td>
                    <td className="p-4 text-sm">
                      {pkg.dayPlans.length} {pkg.dayPlans.length === 1 ? "day" : "days"}
                    </td>
                    <td className="p-4">
                      <Badge variant={pkg.status === "active" ? "default" : "secondary"} className="capitalize">
                        {pkg.status}
                      </Badge>
                    </td>
                    <td className="p-4">
                      <div className="flex justify-end gap-2">
                        <Button
                          variant="ghost"
                          size="icon"
                          onClick={() => onToggleStatus(pkg.id)}
                          title={pkg.status === "active" ? "Deactivate" : "Activate"}
                        >
                          {pkg.status === "active" ? (
                            <ToggleRight className="h-5 w-5 text-green-600" />
                          ) : (
                            <ToggleLeft className="h-5 w-5 text-muted-foreground" />
                          )}
                        </Button>
                        <Button variant="ghost" size="icon" onClick={() => onEdit(pkg.id)} title="Edit package">
                          <Edit className="h-4 w-4" />
                        </Button>
                        <Button
                          variant="ghost"
                          size="icon"
                          onClick={() => onDelete(pkg.id, pkg.cityName)}
                          title="Delete package"
                        >
                          <Trash2 className="h-4 w-4 text-destructive" />
                        </Button>
                      </div>
                    </td>
                  </tr>
                ))
              )}
            </tbody>
          </table>
        </div>

        {totalPages > 1 && (
          <div className="flex items-center justify-between border-t p-4">
            <div className="flex items-center gap-2">
              <span className="text-sm text-muted-foreground">Items per page:</span>
              <Select value={String(itemsPerPage)} onValueChange={(value) => onItemsPerPageChange(Number(value))}>
                <SelectTrigger className="w-20">
                  <SelectValue />
                </SelectTrigger>
                <SelectContent>
                  <SelectItem value="10">10</SelectItem>
                  <SelectItem value="20">20</SelectItem>
                  <SelectItem value="50">50</SelectItem>
                  <SelectItem value="100">100</SelectItem>
                </SelectContent>
              </Select>
            </div>
            <div className="flex items-center gap-2">
              <Button
                variant="outline"
                size="sm"
                onClick={() => onPageChange(currentPage - 1)}
                disabled={currentPage === 1}
              >
                Previous
              </Button>
              <span className="text-sm">
                Page {currentPage} of {totalPages}
              </span>
              <Button
                variant="outline"
                size="sm"
                onClick={() => onPageChange(currentPage + 1)}
                disabled={currentPage === totalPages}
              >
                Next
              </Button>
            </div>
          </div>
        )}
      </Card>

      {selectedPackage && (
        <ViewPackageDetailsDialog
          open={viewDialogOpen}
          onOpenChange={setViewDialogOpen}
          packageData={selectedPackage}
        />
      )}
    </>
  )
}
