"use client"

import { useState, useEffect } from "react"
import { useForm } from "react-hook-form"
import { zodResolver } from "@hookform/resolvers/zod"
import {
  Dialog,
  DialogContent,
  DialogDescription,
  DialogFooter,
  DialogHeader,
  DialogTitle,
} from "@/components/ui/dialog"
import { Button } from "@/components/ui/button"
import { Input } from "@/components/ui/input"
import { Label } from "@/components/ui/label"
import { stateSchema, type StateInput } from "@/lib/validations/location"
import type { State } from "@/lib/data/locations"
import DOMPurify from "dompurify"

interface EditStateDialogProps {
  state: State | null
  open: boolean
  onOpenChange: (open: boolean) => void
  onSave: (id: string, name: string) => boolean
}

export function EditStateDialog({ state, open, onOpenChange, onSave }: EditStateDialogProps) {
  const [isSubmitting, setIsSubmitting] = useState(false)

  const {
    register,
    handleSubmit,
    reset,
    formState: { errors },
  } = useForm<StateInput>({
    resolver: zodResolver(stateSchema),
    defaultValues: { name: "" },
  })

  useEffect(() => {
    if (state) {
      reset({ name: state.name })
    }
  }, [state, reset])

  const onSubmit = async (data: StateInput) => {
    if (!state) return

    setIsSubmitting(true)
    try {
      const sanitizedName = DOMPurify.sanitize(data.name.trim())
      onSave(state.id, sanitizedName)
    } finally {
      setIsSubmitting(false)
    }
  }

  return (
    <Dialog open={open} onOpenChange={onOpenChange}>
      <DialogContent className="sm:max-w-md">
        <DialogHeader>
          <DialogTitle>Edit State</DialogTitle>
          <DialogDescription>Update the state name.</DialogDescription>
        </DialogHeader>
        <form onSubmit={handleSubmit(onSubmit)}>
          <div className="grid gap-4 py-4">
            <div className="grid gap-2">
              <Label htmlFor="edit-state-name">State Name</Label>
              <Input
                id="edit-state-name"
                placeholder="Enter state name"
                {...register("name")}
                aria-invalid={!!errors.name}
                aria-describedby={errors.name ? "edit-state-name-error" : undefined}
              />
              {errors.name && (
                <p id="edit-state-name-error" className="text-sm text-destructive">
                  {errors.name.message}
                </p>
              )}
            </div>
          </div>
          <DialogFooter>
            <Button type="button" variant="outline" onClick={() => onOpenChange(false)}>
              Cancel
            </Button>
            <Button type="submit" disabled={isSubmitting}>
              {isSubmitting ? "Saving..." : "Save Changes"}
            </Button>
          </DialogFooter>
        </form>
      </DialogContent>
    </Dialog>
  )
}
