"use client"

import { useState } from "react"
import { useForm } from "react-hook-form"
import { zodResolver } from "@hookform/resolvers/zod"
import {
  Dialog,
  DialogContent,
  DialogDescription,
  DialogFooter,
  DialogHeader,
  DialogTitle,
} from "@/components/ui/dialog"
import { Button } from "@/components/ui/button"
import { Input } from "@/components/ui/input"
import { Label } from "@/components/ui/label"
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from "@/components/ui/select"
import { vehicleSchema, type VehicleInput } from "@/lib/validations/vehicle"
import type { VehicleCategory } from "@/lib/data/vehicles"
import DOMPurify from "dompurify"

interface AddVehicleDialogProps {
  open: boolean
  onOpenChange: (open: boolean) => void
  categories: VehicleCategory[]
  onAdd: (data: VehicleInput) => boolean
}

export function AddVehicleDialog({ open, onOpenChange, categories, onAdd }: AddVehicleDialogProps) {
  const [isSubmitting, setIsSubmitting] = useState(false)

  const {
    register,
    handleSubmit,
    reset,
    setValue,
    watch,
    formState: { errors },
  } = useForm<VehicleInput>({
    resolver: zodResolver(vehicleSchema),
    defaultValues: {
      name: "",
      categoryId: "",
      photo: "/placeholder.svg?height=80&width=120",
      mileage: 0,
      perKmPrice: 0,
      perDayPrice: 0,
    },
  })

  const selectedCategory = watch("categoryId")

  const onSubmit = async (data: VehicleInput) => {
    setIsSubmitting(true)
    try {
      const sanitizedData: VehicleInput = {
        ...data,
        name: DOMPurify.sanitize(data.name.trim()),
        photo: DOMPurify.sanitize(data.photo.trim()),
      }
      const success = onAdd(sanitizedData)
      if (success) {
        reset()
      }
    } finally {
      setIsSubmitting(false)
    }
  }

  const handleOpenChange = (newOpen: boolean) => {
    if (!newOpen) {
      reset()
    }
    onOpenChange(newOpen)
  }

  return (
    <Dialog open={open} onOpenChange={handleOpenChange}>
      <DialogContent className="sm:max-w-lg max-h-[90vh] flex flex-col">
        <DialogHeader className="flex-shrink-0">
          <DialogTitle>Add New Vehicle</DialogTitle>
          <DialogDescription>Enter the details of the vehicle you want to add.</DialogDescription>
        </DialogHeader>
        <form onSubmit={handleSubmit(onSubmit)} className="flex flex-col flex-1 overflow-hidden">
          <div className="flex-1 overflow-y-auto px-1 py-4 space-y-4">
            <div className="grid gap-2">
              <Label htmlFor="vehicle-name">Vehicle Name</Label>
              <Input
                id="vehicle-name"
                placeholder="Enter vehicle name"
                {...register("name")}
                aria-invalid={!!errors.name}
                aria-describedby={errors.name ? "vehicle-name-error" : undefined}
              />
              {errors.name && (
                <p id="vehicle-name-error" className="text-sm text-destructive">
                  {errors.name.message}
                </p>
              )}
            </div>

            <div className="grid gap-2">
              <Label htmlFor="vehicle-category">Vehicle Category</Label>
              <Select value={selectedCategory} onValueChange={(value) => setValue("categoryId", value)}>
                <SelectTrigger id="vehicle-category" aria-invalid={!!errors.categoryId}>
                  <SelectValue placeholder="Select category" />
                </SelectTrigger>
                <SelectContent>
                  {categories.map((category) => (
                    <SelectItem key={category.id} value={category.id}>
                      {category.name}
                    </SelectItem>
                  ))}
                </SelectContent>
              </Select>
              {errors.categoryId && <p className="text-sm text-destructive">{errors.categoryId.message}</p>}
            </div>

            <div className="grid gap-2">
              <Label htmlFor="vehicle-photo">Photo URL</Label>
              <Input
                id="vehicle-photo"
                placeholder="Enter photo URL"
                {...register("photo")}
                aria-invalid={!!errors.photo}
                aria-describedby={errors.photo ? "vehicle-photo-error" : undefined}
              />
              {errors.photo && (
                <p id="vehicle-photo-error" className="text-sm text-destructive">
                  {errors.photo.message}
                </p>
              )}
            </div>

            <div className="grid gap-2">
              <Label htmlFor="vehicle-mileage">Mileage (km/l)</Label>
              <Input
                id="vehicle-mileage"
                type="number"
                placeholder="Enter mileage"
                {...register("mileage", { valueAsNumber: true })}
                aria-invalid={!!errors.mileage}
                aria-describedby={errors.mileage ? "vehicle-mileage-error" : undefined}
              />
              {errors.mileage && (
                <p id="vehicle-mileage-error" className="text-sm text-destructive">
                  {errors.mileage.message}
                </p>
              )}
            </div>

            <div className="grid gap-2">
              <Label htmlFor="vehicle-per-km">Per KM Price (₹)</Label>
              <Input
                id="vehicle-per-km"
                type="number"
                placeholder="Enter per KM price"
                {...register("perKmPrice", { valueAsNumber: true })}
                aria-invalid={!!errors.perKmPrice}
                aria-describedby={errors.perKmPrice ? "vehicle-per-km-error" : undefined}
              />
              {errors.perKmPrice && (
                <p id="vehicle-per-km-error" className="text-sm text-destructive">
                  {errors.perKmPrice.message}
                </p>
              )}
            </div>

            <div className="grid gap-2">
              <Label htmlFor="vehicle-per-day">Per Day Price (₹)</Label>
              <Input
                id="vehicle-per-day"
                type="number"
                placeholder="Enter per day price"
                {...register("perDayPrice", { valueAsNumber: true })}
                aria-invalid={!!errors.perDayPrice}
                aria-describedby={errors.perDayPrice ? "vehicle-per-day-error" : undefined}
              />
              {errors.perDayPrice && (
                <p id="vehicle-per-day-error" className="text-sm text-destructive">
                  {errors.perDayPrice.message}
                </p>
              )}
            </div>
          </div>
          <DialogFooter className="flex-shrink-0 border-t pt-4">
            <Button type="button" variant="outline" onClick={() => handleOpenChange(false)}>
              Cancel
            </Button>
            <Button type="submit" disabled={isSubmitting}>
              {isSubmitting ? "Adding..." : "Add Vehicle"}
            </Button>
          </DialogFooter>
        </form>
      </DialogContent>
    </Dialog>
  )
}
