"use client"

import { useState, useEffect } from "react"
import { useForm } from "react-hook-form"
import { zodResolver } from "@hookform/resolvers/zod"
import {
  Dialog,
  DialogContent,
  DialogDescription,
  DialogFooter,
  DialogHeader,
  DialogTitle,
} from "@/components/ui/dialog"
import { Button } from "@/components/ui/button"
import { Input } from "@/components/ui/input"
import { Label } from "@/components/ui/label"
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from "@/components/ui/select"
import { vehicleSchema, type VehicleInput } from "@/lib/validations/vehicle"
import type { Vehicle, VehicleCategory } from "@/lib/data/vehicles"
import DOMPurify from "dompurify"

interface EditVehicleDialogProps {
  vehicle: Vehicle | null
  open: boolean
  onOpenChange: (open: boolean) => void
  categories: VehicleCategory[]
  onSave: (id: string, data: VehicleInput) => boolean
}

export function EditVehicleDialog({ vehicle, open, onOpenChange, categories, onSave }: EditVehicleDialogProps) {
  const [isSubmitting, setIsSubmitting] = useState(false)

  const {
    register,
    handleSubmit,
    reset,
    setValue,
    watch,
    formState: { errors },
  } = useForm<VehicleInput>({
    resolver: zodResolver(vehicleSchema),
    defaultValues: {
      name: "",
      categoryId: "",
      photo: "",
      mileage: 0,
      perKmPrice: 0,
      perDayPrice: 0,
    },
  })

  const selectedCategory = watch("categoryId")

  useEffect(() => {
    if (vehicle) {
      reset({
        name: vehicle.name,
        categoryId: vehicle.categoryId,
        photo: vehicle.photo,
        mileage: vehicle.mileage,
        perKmPrice: vehicle.perKmPrice,
        perDayPrice: vehicle.perDayPrice,
      })
    }
  }, [vehicle, reset])

  const onSubmit = async (data: VehicleInput) => {
    if (!vehicle) return

    setIsSubmitting(true)
    try {
      const sanitizedData: VehicleInput = {
        ...data,
        name: DOMPurify.sanitize(data.name.trim()),
        photo: DOMPurify.sanitize(data.photo.trim()),
      }
      onSave(vehicle.id, sanitizedData)
    } finally {
      setIsSubmitting(false)
    }
  }

  return (
    <Dialog open={open} onOpenChange={onOpenChange}>
      <DialogContent className="sm:max-w-lg max-h-[90vh] flex flex-col">
        <DialogHeader className="flex-shrink-0">
          <DialogTitle>Edit Vehicle</DialogTitle>
          <DialogDescription>Update the vehicle details.</DialogDescription>
        </DialogHeader>
        <form onSubmit={handleSubmit(onSubmit)} className="flex flex-col flex-1 overflow-hidden">
          <div className="flex-1 overflow-y-auto px-1 py-4 space-y-4">
            <div className="grid gap-2">
              <Label htmlFor="edit-vehicle-name">Vehicle Name</Label>
              <Input
                id="edit-vehicle-name"
                placeholder="Enter vehicle name"
                {...register("name")}
                aria-invalid={!!errors.name}
                aria-describedby={errors.name ? "edit-vehicle-name-error" : undefined}
              />
              {errors.name && (
                <p id="edit-vehicle-name-error" className="text-sm text-destructive">
                  {errors.name.message}
                </p>
              )}
            </div>

            <div className="grid gap-2">
              <Label htmlFor="edit-vehicle-category">Vehicle Category</Label>
              <Select value={selectedCategory} onValueChange={(value) => setValue("categoryId", value)}>
                <SelectTrigger id="edit-vehicle-category" aria-invalid={!!errors.categoryId}>
                  <SelectValue placeholder="Select category" />
                </SelectTrigger>
                <SelectContent>
                  {categories.map((category) => (
                    <SelectItem key={category.id} value={category.id}>
                      {category.name}
                    </SelectItem>
                  ))}
                </SelectContent>
              </Select>
              {errors.categoryId && <p className="text-sm text-destructive">{errors.categoryId.message}</p>}
            </div>

            <div className="grid gap-2">
              <Label htmlFor="edit-vehicle-photo">Photo URL</Label>
              <Input
                id="edit-vehicle-photo"
                placeholder="Enter photo URL"
                {...register("photo")}
                aria-invalid={!!errors.photo}
                aria-describedby={errors.photo ? "edit-vehicle-photo-error" : undefined}
              />
              {errors.photo && (
                <p id="edit-vehicle-photo-error" className="text-sm text-destructive">
                  {errors.photo.message}
                </p>
              )}
            </div>

            <div className="grid gap-2">
              <Label htmlFor="edit-vehicle-mileage">Mileage (km/l)</Label>
              <Input
                id="edit-vehicle-mileage"
                type="number"
                placeholder="Enter mileage"
                {...register("mileage", { valueAsNumber: true })}
                aria-invalid={!!errors.mileage}
                aria-describedby={errors.mileage ? "edit-vehicle-mileage-error" : undefined}
              />
              {errors.mileage && (
                <p id="edit-vehicle-mileage-error" className="text-sm text-destructive">
                  {errors.mileage.message}
                </p>
              )}
            </div>

            <div className="grid gap-2">
              <Label htmlFor="edit-vehicle-per-km">Per KM Price (₹)</Label>
              <Input
                id="edit-vehicle-per-km"
                type="number"
                placeholder="Enter per KM price"
                {...register("perKmPrice", { valueAsNumber: true })}
                aria-invalid={!!errors.perKmPrice}
                aria-describedby={errors.perKmPrice ? "edit-vehicle-per-km-error" : undefined}
              />
              {errors.perKmPrice && (
                <p id="edit-vehicle-per-km-error" className="text-sm text-destructive">
                  {errors.perKmPrice.message}
                </p>
              )}
            </div>

            <div className="grid gap-2">
              <Label htmlFor="edit-vehicle-per-day">Per Day Price (₹)</Label>
              <Input
                id="edit-vehicle-per-day"
                type="number"
                placeholder="Enter per day price"
                {...register("perDayPrice", { valueAsNumber: true })}
                aria-invalid={!!errors.perDayPrice}
                aria-describedby={errors.perDayPrice ? "edit-vehicle-per-day-error" : undefined}
              />
              {errors.perDayPrice && (
                <p id="edit-vehicle-per-day-error" className="text-sm text-destructive">
                  {errors.perDayPrice.message}
                </p>
              )}
            </div>
          </div>
          <DialogFooter className="flex-shrink-0 border-t pt-4">
            <Button type="button" variant="outline" onClick={() => onOpenChange(false)}>
              Cancel
            </Button>
            <Button type="submit" disabled={isSubmitting}>
              {isSubmitting ? "Saving..." : "Save Changes"}
            </Button>
          </DialogFooter>
        </form>
      </DialogContent>
    </Dialog>
  )
}
