"use client"

import { useState, useMemo, useCallback } from "react"
import { Tabs, TabsContent, TabsList, TabsTrigger } from "@/components/ui/tabs"
import { Button } from "@/components/ui/button"
import { Plus } from "lucide-react"
import { VehicleTable } from "./vehicle-table"
import { AddVehicleDialog } from "./add-vehicle-dialog"
import { EditVehicleDialog } from "./edit-vehicle-dialog"
import { DeleteVehicleDialog } from "./delete-vehicle-dialog"
import { vehicleDataStore, type Vehicle } from "@/lib/data/vehicles"
import type { VehicleInput } from "@/lib/validations/vehicle"
import { useToastEnhanced } from "@/hooks/use-toast-enhanced"

export function VehiclesClient() {
  const toast = useToastEnhanced()
  const [currentTab, setCurrentTab] = useState("all")

  const [vehicles, setVehicles] = useState<Vehicle[]>(vehicleDataStore.getAllVehicles())
  const [categories] = useState(vehicleDataStore.getActiveCategories())
  const [isAddDialogOpen, setIsAddDialogOpen] = useState(false)
  const [editingVehicle, setEditingVehicle] = useState<Vehicle | null>(null)
  const [deletingVehicle, setDeletingVehicle] = useState<Vehicle | null>(null)

  const activeVehicles = useMemo(() => vehicles.filter((v) => v.status === "active"), [vehicles])
  const inactiveVehicles = useMemo(() => vehicles.filter((v) => v.status === "inactive"), [vehicles])

  const handleTabChange = (value: string) => {
    setCurrentTab(value)
  }

  const handleAddVehicle = useCallback(
    (data: VehicleInput) => {
      const newVehicle = vehicleDataStore.addVehicle({
        name: data.name,
        categoryId: data.categoryId,
        photo: data.photo,
        mileage: data.mileage,
        perKmPrice: data.perKmPrice,
        perDayPrice: data.perDayPrice,
      })
      if (newVehicle) {
        setVehicles(vehicleDataStore.getAllVehicles())
        toast.success({
          title: "Vehicle Added",
          description: `${data.name} has been added successfully.`,
        })
        setIsAddDialogOpen(false)
        return true
      }
      toast.error({
        title: "Failed to Add Vehicle",
        description: "A vehicle with this name already exists.",
      })
      return false
    },
    [toast],
  )

  const handleEditVehicle = useCallback(
    (id: string, data: VehicleInput) => {
      const updated = vehicleDataStore.updateVehicle(id, {
        name: data.name,
        categoryId: data.categoryId,
        photo: data.photo,
        mileage: data.mileage,
        perKmPrice: data.perKmPrice,
        perDayPrice: data.perDayPrice,
      })
      if (updated) {
        setVehicles(vehicleDataStore.getAllVehicles())
        toast.success({
          title: "Vehicle Updated",
          description: `${data.name} has been updated successfully.`,
        })
        setEditingVehicle(null)
        return true
      }
      toast.error({
        title: "Failed to Update Vehicle",
        description: "A vehicle with this name already exists.",
      })
      return false
    },
    [toast],
  )

  const handleToggleStatus = useCallback(
    (vehicle: Vehicle) => {
      const updated = vehicleDataStore.toggleVehicleStatus(vehicle.id)
      if (updated) {
        setVehicles(vehicleDataStore.getAllVehicles())
        const action = updated.status === "active" ? "activated" : "deactivated"
        toast.success({
          title: `Vehicle ${action.charAt(0).toUpperCase() + action.slice(1)}`,
          description: `${vehicle.name} has been ${action}.`,
        })
      }
    },
    [toast],
  )

  const handleDeleteVehicle = useCallback(() => {
    if (!deletingVehicle) return

    const deleted = vehicleDataStore.deleteVehicle(deletingVehicle.id)
    if (deleted) {
      setVehicles(vehicleDataStore.getAllVehicles())
      toast.success({
        title: "Vehicle Deleted",
        description: `${deletingVehicle.name} has been deleted.`,
      })
    }
    setDeletingVehicle(null)
  }, [deletingVehicle, toast])

  const getDisplayVehicles = () => {
    switch (currentTab) {
      case "active":
        return activeVehicles
      case "inactive":
        return inactiveVehicles
      default:
        return vehicles
    }
  }

  return (
    <section aria-labelledby="vehicles-heading" className="admin-section">
      <header className="admin-page-header flex flex-col gap-4 sm:flex-row sm:items-center sm:justify-between">
        <div>
          <h1 id="vehicles-heading" className="admin-page-title">
            Vehicle Listing
          </h1>
          <p className="admin-page-description">Manage all vehicles available for booking.</p>
        </div>
        <Button onClick={() => setIsAddDialogOpen(true)} className="w-full sm:w-auto">
          <Plus className="mr-2 h-4 w-4" aria-hidden="true" />
          Add Vehicle
        </Button>
      </header>

      <Tabs value={currentTab} onValueChange={handleTabChange} className="w-full">
        <TabsList className="mb-6 h-auto p-1.5 bg-muted/50 rounded-xl border border-border/50">
          <TabsTrigger
            value="all"
            className="rounded-lg data-[state=active]:bg-background data-[state=active]:shadow-sm px-4 py-2"
          >
            All ({vehicles.length})
          </TabsTrigger>
          <TabsTrigger
            value="active"
            className="rounded-lg data-[state=active]:bg-background data-[state=active]:shadow-sm px-4 py-2"
          >
            Active ({activeVehicles.length})
          </TabsTrigger>
          <TabsTrigger
            value="inactive"
            className="rounded-lg data-[state=active]:bg-background data-[state=active]:shadow-sm px-4 py-2"
          >
            Inactive ({inactiveVehicles.length})
          </TabsTrigger>
        </TabsList>

        <TabsContent value="all" className="mt-0">
          <VehicleTable
            vehicles={getDisplayVehicles()}
            onEdit={setEditingVehicle}
            onDelete={setDeletingVehicle}
            onToggleStatus={handleToggleStatus}
          />
        </TabsContent>

        <TabsContent value="active" className="mt-0">
          <VehicleTable
            vehicles={getDisplayVehicles()}
            onEdit={setEditingVehicle}
            onDelete={setDeletingVehicle}
            onToggleStatus={handleToggleStatus}
          />
        </TabsContent>

        <TabsContent value="inactive" className="mt-0">
          <VehicleTable
            vehicles={getDisplayVehicles()}
            onEdit={setEditingVehicle}
            onDelete={setDeletingVehicle}
            onToggleStatus={handleToggleStatus}
          />
        </TabsContent>
      </Tabs>

      <AddVehicleDialog
        open={isAddDialogOpen}
        onOpenChange={setIsAddDialogOpen}
        categories={categories}
        onAdd={handleAddVehicle}
      />

      <EditVehicleDialog
        vehicle={editingVehicle}
        open={!!editingVehicle}
        onOpenChange={(open) => !open && setEditingVehicle(null)}
        categories={categories}
        onSave={handleEditVehicle}
      />

      <DeleteVehicleDialog
        vehicle={deletingVehicle}
        open={!!deletingVehicle}
        onOpenChange={(open) => !open && setDeletingVehicle(null)}
        onConfirm={handleDeleteVehicle}
      />
    </section>
  )
}
