"use client"

import Link from "next/link"
import { Button } from "@/components/ui/button"
import { Car, Hotel, Sparkles, Package, LogIn, UserPlus, Menu, X } from "lucide-react"
import { useState } from "react"
import { cn } from "@/lib/utils"

export function FrontendHeader() {
  const [mobileMenuOpen, setMobileMenuOpen] = useState(false)

  const navLinks = [
    { href: "/vehicle-booking", label: "Vehicle Booking", icon: Car },
    { href: "/hotel-booking", label: "Hotel Booking", icon: Hotel },
    { href: "/spiritual-packages", label: "Spiritual Packages", icon: Sparkles },
    { href: "/custom-packages", label: "Custom Packages", icon: Package },
  ]

  return (
    <header className="sticky top-0 z-50 w-full border-b border-border/40 bg-background shadow-sm">
      <nav
        className="mx-auto flex h-14 max-w-7xl items-center justify-between px-4 md:px-6 lg:px-8"
        aria-label="Main navigation"
      >
        {/* Logo */}
        <Link href="/" className="flex items-center space-x-2 group">
          <div className="relative flex h-8 w-8 items-center justify-center rounded-lg frontend-gradient-primary shadow-md group-hover:scale-110 transition-transform">
            <Sparkles className="h-4 w-4 text-white" aria-hidden="true" />
          </div>
          <span className="text-xl font-bold frontend-text-gradient">Teertham</span>
        </Link>

        {/* Desktop Navigation */}
        <div className="hidden lg:flex items-center gap-1">
          {navLinks.map((link) => {
            const Icon = link.icon
            return (
              <Button
                key={link.href}
                asChild
                variant="ghost"
                size="sm"
                className="gap-1.5 hover:bg-primary/10 hover:text-primary transition-colors"
              >
                <Link href={link.href}>
                  <Icon className="h-4 w-4" aria-hidden="true" />
                  <span className="font-medium text-sm">{link.label}</span>
                </Link>
              </Button>
            )
          })}
        </div>

        {/* Auth Buttons */}
        <div className="hidden md:flex items-center gap-2">
          <Button
            asChild
            variant="outline"
            size="sm"
            className="gap-1.5 hover:bg-primary/10 hover:text-primary hover:border-primary transition-all bg-transparent"
            aria-label="Login to your account"
          >
            <Link href="/login">
              <LogIn className="h-4 w-4" aria-hidden="true" />
              <span>Login</span>
            </Link>
          </Button>

          <Button
            asChild
            size="sm"
            className="gap-1.5 frontend-gradient-primary hover:opacity-90 transition-opacity shadow-md"
            aria-label="Create a new account"
          >
            <Link href="/create-account">
              <UserPlus className="h-4 w-4" aria-hidden="true" />
              <span>Create Account</span>
            </Link>
          </Button>
        </div>

        {/* Mobile Menu Button */}
        <Button
          variant="ghost"
          size="icon"
          className="lg:hidden h-8 w-8"
          onClick={() => setMobileMenuOpen(!mobileMenuOpen)}
          aria-label="Toggle mobile menu"
          aria-expanded={mobileMenuOpen}
        >
          {mobileMenuOpen ? (
            <X className="h-5 w-5" aria-hidden="true" />
          ) : (
            <Menu className="h-5 w-5" aria-hidden="true" />
          )}
        </Button>
      </nav>

      {/* Mobile Menu - solid background */}
      <div
        className={cn(
          "lg:hidden border-t border-border/40 bg-background overflow-hidden transition-all duration-300",
          mobileMenuOpen ? "max-h-[600px] opacity-100" : "max-h-0 opacity-0",
        )}
      >
        <div className="px-4 py-4 space-y-2">
          {navLinks.map((link) => {
            const Icon = link.icon
            return (
              <Button
                key={link.href}
                asChild
                variant="ghost"
                className="w-full justify-start gap-2 h-10 hover:bg-primary/10 hover:text-primary"
                onClick={() => setMobileMenuOpen(false)}
              >
                <Link href={link.href}>
                  <Icon className="h-4 w-4" aria-hidden="true" />
                  <span className="font-medium">{link.label}</span>
                </Link>
              </Button>
            )
          })}

          <div className="pt-3 space-y-2 border-t border-border/40">
            <Button
              asChild
              variant="outline"
              className="w-full gap-2 h-10 bg-transparent"
              onClick={() => setMobileMenuOpen(false)}
            >
              <Link href="/login">
                <LogIn className="h-4 w-4" aria-hidden="true" />
                <span>Login</span>
              </Link>
            </Button>

            <Button
              asChild
              className="w-full gap-2 h-10 frontend-gradient-primary"
              onClick={() => setMobileMenuOpen(false)}
            >
              <Link href="/create-account">
                <UserPlus className="h-4 w-4" aria-hidden="true" />
                <span>Create Account</span>
              </Link>
            </Button>
          </div>
        </div>
      </div>
    </header>
  )
}
