"use client"

import { useState, useEffect, useCallback } from "react"
import { Button } from "@/components/ui/button"
import { Badge } from "@/components/ui/badge"
import { ChevronLeft, ChevronRight, ArrowRight, Sparkles } from "lucide-react"
import Link from "next/link"
import Image from "next/image"
import { cn } from "@/lib/utils"

const slides = [
  {
    id: 1,
    image: "/sacred-ganges-river-varanasi-india-evening-aarti-c.jpg",
    title: "Discover Sacred India",
    subtitle: "Spiritual Journeys",
    description:
      "Experience divine blessings at India's most sacred destinations. From Varanasi to Rishikesh, embark on a spiritual awakening.",
    cta: "Explore Packages",
    href: "/spiritual-packages",
  },
  {
    id: 2,
    image: "/ancient-hindu-temple-in-himalayan-mountains-with-s.jpg",
    title: "Divine Temples Await",
    subtitle: "Sacred Destinations",
    description: "Visit ancient temples nestled in the Himalayas. Experience peace and spirituality like never before.",
    cta: "Book Now",
    href: "/spiritual-packages",
  },
  {
    id: 3,
    image: "/char-dham-pilgrimage-kedarnath-temple-snow-covered.jpg",
    title: "Char Dham Yatra",
    subtitle: "Holy Pilgrimage",
    description: "Embark on the sacred Char Dham pilgrimage. Book vehicles, hotels, and complete packages with ease.",
    cta: "Plan Your Yatra",
    href: "/custom-packages",
  },
]

export function HeroSlider() {
  const [currentSlide, setCurrentSlide] = useState(0)
  const [isAutoPlaying, setIsAutoPlaying] = useState(true)

  const nextSlide = useCallback(() => {
    setCurrentSlide((prev) => (prev + 1) % slides.length)
  }, [])

  const prevSlide = useCallback(() => {
    setCurrentSlide((prev) => (prev - 1 + slides.length) % slides.length)
  }, [])

  const goToSlide = useCallback((index: number) => {
    setCurrentSlide(index)
    setIsAutoPlaying(false)
    setTimeout(() => setIsAutoPlaying(true), 5000)
  }, [])

  useEffect(() => {
    if (!isAutoPlaying) return
    const interval = setInterval(nextSlide, 5000)
    return () => clearInterval(interval)
  }, [isAutoPlaying, nextSlide])

  return (
    <section className="relative h-[100vh] w-full overflow-hidden">
      {/* Slides */}
      {slides.map((slide, index) => (
        <div
          key={slide.id}
          className={cn(
            "absolute inset-0 transition-all duration-700 ease-in-out",
            index === currentSlide ? "opacity-100 scale-100" : "opacity-0 scale-105",
          )}
        >
          <Image
            src={slide.image || "/placeholder.svg"}
            alt={slide.title}
            fill
            className="object-cover"
            priority={index === 0}
            loading={index === 0 ? "eager" : "lazy"}
          />

          {/* Gradient Overlay */}
          <div className="absolute inset-0 bg-gradient-to-r from-black/70 via-black/50 to-transparent" />
          <div className="absolute inset-0 bg-gradient-to-t from-black/60 via-transparent to-transparent" />

          {/* Content */}
          <div className="relative h-full flex items-center">
            <div className="mx-auto max-w-7xl px-4 md:px-6 lg:px-8 w-full">
              <div
                className={cn(
                  "max-w-2xl space-y-6 transition-all duration-700 delay-200",
                  index === currentSlide ? "translate-y-0 opacity-100" : "translate-y-8 opacity-0",
                )}
              >
                <Badge className="bg-white/20 text-white border-white/30 backdrop-blur-sm px-4 py-2">
                  <Sparkles className="h-4 w-4 mr-2" />
                  {slide.subtitle}
                </Badge>

                <h1 className="text-4xl md:text-6xl lg:text-7xl font-bold text-white leading-tight text-balance">
                  {slide.title}
                </h1>

                <p className="text-lg md:text-xl text-white/90 max-w-xl leading-relaxed">{slide.description}</p>

                <div className="flex flex-wrap gap-4 pt-4">
                  <Button
                    size="lg"
                    asChild
                    className="text-base px-8 h-14 rounded-xl frontend-gradient-primary shadow-xl hover:opacity-90 transition-all hover:scale-105"
                  >
                    <Link href={slide.href}>
                      {slide.cta}
                      <ArrowRight className="ml-2 h-5 w-5" />
                    </Link>
                  </Button>
                  <Button
                    size="lg"
                    variant="outline"
                    asChild
                    className="text-base px-8 h-14 rounded-xl bg-white/10 border-white/30 text-white hover:bg-white/20 transition-all backdrop-blur-sm"
                  >
                    <Link href="/create-account">Get Started Free</Link>
                  </Button>
                </div>
              </div>
            </div>
          </div>
        </div>
      ))}

      {/* Navigation Arrows */}
      <button
        onClick={prevSlide}
        className="absolute left-4 md:left-8 top-1/2 -translate-y-1/2 h-12 w-12 rounded-full bg-white/20 backdrop-blur-sm border border-white/30 flex items-center justify-center text-white hover:bg-white/30 transition-all hover:scale-110 z-10"
        aria-label="Previous slide"
      >
        <ChevronLeft className="h-6 w-6" />
      </button>
      <button
        onClick={nextSlide}
        className="absolute right-4 md:right-8 top-1/2 -translate-y-1/2 h-12 w-12 rounded-full bg-white/20 backdrop-blur-sm border border-white/30 flex items-center justify-center text-white hover:bg-white/30 transition-all hover:scale-110 z-10"
        aria-label="Next slide"
      >
        <ChevronRight className="h-6 w-6" />
      </button>

      {/* Slide Indicators */}
      <div className="absolute bottom-8 left-1/2 -translate-x-1/2 flex items-center gap-3 z-10">
        {slides.map((_, index) => (
          <button
            key={index}
            onClick={() => goToSlide(index)}
            className={cn(
              "h-2 rounded-full transition-all duration-300",
              index === currentSlide ? "w-10 bg-white" : "w-2 bg-white/50 hover:bg-white/70",
            )}
            aria-label={`Go to slide ${index + 1}`}
          />
        ))}
      </div>

      {/* Scroll Indicator */}
      <div className="absolute bottom-8 right-8 hidden md:flex items-center gap-2 text-white/70 text-sm z-10">
        <span>Scroll to explore</span>
        <div className="w-6 h-10 rounded-full border-2 border-white/30 flex justify-center pt-2">
          <div className="w-1 h-2 bg-white/70 rounded-full animate-bounce" />
        </div>
      </div>
    </section>
  )
}
