"use client"

import { useState } from "react"
import { Card, CardContent } from "@/components/ui/card"
import { Button } from "@/components/ui/button"
import { Input } from "@/components/ui/input"
import { Label } from "@/components/ui/label"
import {
  Dialog,
  DialogContent,
  DialogHeader,
  DialogTitle,
  DialogDescription,
  DialogFooter,
} from "@/components/ui/dialog"
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from "@/components/ui/select"
import { Badge } from "@/components/ui/badge"
import {
  Wallet,
  ArrowDownToLine,
  Clock,
  IndianRupee,
  CreditCard,
  Plus,
  Edit2,
  Trash2,
  Building,
  CheckCircle,
  AlertCircle,
} from "lucide-react"

interface PaymentMethod {
  id: string
  type: "bank" | "upi"
  bankName?: string
  accountNumber?: string
  ifscCode?: string
  accountHolder?: string
  upiId?: string
  isPrimary: boolean
}

const initialPaymentMethods: PaymentMethod[] = [
  {
    id: "1",
    type: "bank",
    bankName: "HDFC Bank",
    accountNumber: "XXXX XXXX 4521",
    ifscCode: "HDFC0001234",
    accountHolder: "Grand Palace Hotels Pvt Ltd",
    isPrimary: true,
  },
  {
    id: "2",
    type: "bank",
    bankName: "ICICI Bank",
    accountNumber: "XXXX XXXX 7832",
    ifscCode: "ICIC0005678",
    accountHolder: "Grand Palace Hotels Pvt Ltd",
    isPrimary: false,
  },
  {
    id: "3",
    type: "upi",
    upiId: "grandpalace@upi",
    isPrimary: false,
  },
]

const summaryData = [
  {
    title: "Total Earnings",
    value: "₹12,45,000",
    change: "+18.5% from last month",
    icon: IndianRupee,
    color: "bg-emerald-500",
  },
  {
    title: "Available Balance",
    value: "₹2,85,000",
    subtitle: "Ready to withdraw",
    icon: Wallet,
    color: "bg-blue-500",
  },
  {
    title: "Pending Withdrawals",
    value: "₹73,500",
    subtitle: "Processing",
    icon: Clock,
    color: "bg-amber-500",
  },
  {
    title: "Total Withdrawn",
    value: "₹8,86,500",
    subtitle: "This year",
    icon: ArrowDownToLine,
    color: "bg-violet-500",
  },
]

export function EarningSummary() {
  const [paymentMethods, setPaymentMethods] = useState<PaymentMethod[]>(initialPaymentMethods)
  const [isMethodsDialogOpen, setIsMethodsDialogOpen] = useState(false)
  const [isAddMethodDialogOpen, setIsAddMethodDialogOpen] = useState(false)
  const [isWithdrawDialogOpen, setIsWithdrawDialogOpen] = useState(false)
  const [isEditMethodDialogOpen, setIsEditMethodDialogOpen] = useState(false)
  const [selectedMethod, setSelectedMethod] = useState<PaymentMethod | null>(null)

  const [methodForm, setMethodForm] = useState({
    type: "bank" as "bank" | "upi",
    bankName: "",
    accountNumber: "",
    ifscCode: "",
    accountHolder: "",
    upiId: "",
  })

  const [withdrawForm, setWithdrawForm] = useState({
    amount: "",
    methodId: "",
  })

  const resetMethodForm = () => {
    setMethodForm({
      type: "bank",
      bankName: "",
      accountNumber: "",
      ifscCode: "",
      accountHolder: "",
      upiId: "",
    })
  }

  const handleAddMethod = () => {
    const newMethod: PaymentMethod = {
      id: Date.now().toString(),
      type: methodForm.type,
      bankName: methodForm.type === "bank" ? methodForm.bankName : undefined,
      accountNumber: methodForm.type === "bank" ? methodForm.accountNumber : undefined,
      ifscCode: methodForm.type === "bank" ? methodForm.ifscCode : undefined,
      accountHolder: methodForm.type === "bank" ? methodForm.accountHolder : undefined,
      upiId: methodForm.type === "upi" ? methodForm.upiId : undefined,
      isPrimary: paymentMethods.length === 0,
    }
    setPaymentMethods([...paymentMethods, newMethod])
    setIsAddMethodDialogOpen(false)
    resetMethodForm()
  }

  const handleEditMethod = () => {
    if (!selectedMethod) return
    setPaymentMethods(
      paymentMethods.map((method) =>
        method.id === selectedMethod.id
          ? {
              ...method,
              type: methodForm.type,
              bankName: methodForm.type === "bank" ? methodForm.bankName : undefined,
              accountNumber: methodForm.type === "bank" ? methodForm.accountNumber : undefined,
              ifscCode: methodForm.type === "bank" ? methodForm.ifscCode : undefined,
              accountHolder: methodForm.type === "bank" ? methodForm.accountHolder : undefined,
              upiId: methodForm.type === "upi" ? methodForm.upiId : undefined,
            }
          : method,
      ),
    )
    setIsEditMethodDialogOpen(false)
    setSelectedMethod(null)
    resetMethodForm()
  }

  const handleDeleteMethod = (id: string) => {
    setPaymentMethods(paymentMethods.filter((method) => method.id !== id))
  }

  const handleSetPrimary = (id: string) => {
    setPaymentMethods(paymentMethods.map((method) => ({ ...method, isPrimary: method.id === id })))
  }

  const openEditMethod = (method: PaymentMethod) => {
    setSelectedMethod(method)
    setMethodForm({
      type: method.type,
      bankName: method.bankName || "",
      accountNumber: method.accountNumber?.replace(/X/g, "") || "",
      ifscCode: method.ifscCode || "",
      accountHolder: method.accountHolder || "",
      upiId: method.upiId || "",
    })
    setIsEditMethodDialogOpen(true)
  }

  const handleWithdrawSubmit = () => {
    console.log("Withdraw request:", withdrawForm)
    setIsWithdrawDialogOpen(false)
    setWithdrawForm({ amount: "", methodId: "" })
  }

  const primaryMethod = paymentMethods.find((m) => m.isPrimary)

  return (
    <div className="space-y-4">
      {/* Summary Cards */}
      <div className="grid grid-cols-1 sm:grid-cols-2 lg:grid-cols-4 gap-4">
        {summaryData.map((item) => {
          const Icon = item.icon
          return (
            <Card key={item.title} className="border bg-card">
              <CardContent className="p-5">
                <div className="flex items-start justify-between">
                  <div className="space-y-1">
                    <p className="text-sm text-muted-foreground">{item.title}</p>
                    <p className="text-2xl font-bold">{item.value}</p>
                    <p className="text-xs text-muted-foreground">{item.change || item.subtitle}</p>
                  </div>
                  <div className={`p-2.5 rounded-xl ${item.color}`}>
                    <Icon className="h-5 w-5 text-white" />
                  </div>
                </div>
              </CardContent>
            </Card>
          )
        })}
      </div>

      {/* Action Card */}
      <Card className="border bg-gradient-to-r from-slate-900 to-slate-800 dark:from-slate-800 dark:to-slate-700 text-white">
        <CardContent className="p-6">
          <div className="flex flex-col md:flex-row md:items-center md:justify-between gap-4">
            <div className="flex items-center gap-4">
              <div className="p-3 rounded-xl bg-white/10">
                <CreditCard className="h-6 w-6" />
              </div>
              <div>
                <h3 className="font-semibold text-lg">Ready to Withdraw?</h3>
                <p className="text-sm text-white/70">Transfer your earnings to your bank account</p>
              </div>
            </div>
            <div className="flex flex-col sm:flex-row items-stretch sm:items-center gap-3">
              <div className="text-right">
                <p className="text-xs text-white/60">Available Balance</p>
                <p className="text-2xl font-bold">₹2,85,000</p>
              </div>
              <div className="flex gap-2">
                <Button
                  variant="outline"
                  className="bg-transparent border-white/30 text-white hover:bg-white/10 hover:text-white"
                  onClick={() => setIsMethodsDialogOpen(true)}
                >
                  <CreditCard className="h-4 w-4 mr-2" />
                  Withdraw Methods
                </Button>
                <Button
                  className="bg-white text-slate-900 hover:bg-white/90 shadow-lg"
                  onClick={() => setIsWithdrawDialogOpen(true)}
                >
                  <ArrowDownToLine className="h-4 w-4 mr-2" />
                  Withdraw Now
                </Button>
              </div>
            </div>
          </div>
        </CardContent>
      </Card>

      {/* Withdraw Methods Dialog */}
      <Dialog open={isMethodsDialogOpen} onOpenChange={setIsMethodsDialogOpen}>
        <DialogContent className="max-w-xl">
          <DialogHeader>
            <DialogTitle className="flex items-center gap-2">
              <CreditCard className="h-5 w-5" />
              Withdraw Methods
            </DialogTitle>
            <DialogDescription>Manage your payment methods for withdrawals</DialogDescription>
          </DialogHeader>

          <div className="space-y-4 py-4">
            {paymentMethods.length === 0 ? (
              <div className="text-center py-8">
                <CreditCard className="h-12 w-12 mx-auto text-muted-foreground/50 mb-3" />
                <p className="text-muted-foreground">No payment methods added yet</p>
              </div>
            ) : (
              <div className="space-y-3">
                {paymentMethods.map((method) => (
                  <div
                    key={method.id}
                    className={`p-4 rounded-xl border ${method.isPrimary ? "border-emerald-500 bg-emerald-50 dark:bg-emerald-950/30" : "border-border"}`}
                  >
                    <div className="flex items-start justify-between">
                      <div className="flex items-center gap-3">
                        <div className="p-2 rounded-lg bg-slate-100 dark:bg-slate-800">
                          {method.type === "bank" ? (
                            <Building className="h-5 w-5 text-slate-600 dark:text-slate-300" />
                          ) : (
                            <Wallet className="h-5 w-5 text-slate-600 dark:text-slate-300" />
                          )}
                        </div>
                        <div>
                          {method.type === "bank" ? (
                            <>
                              <p className="font-medium">{method.bankName}</p>
                              <p className="text-sm text-muted-foreground">{method.accountNumber}</p>
                              <p className="text-xs text-muted-foreground">{method.accountHolder}</p>
                            </>
                          ) : (
                            <>
                              <p className="font-medium">UPI</p>
                              <p className="text-sm text-muted-foreground">{method.upiId}</p>
                            </>
                          )}
                        </div>
                      </div>
                      <div className="flex items-center gap-2">
                        {method.isPrimary && (
                          <Badge className="bg-emerald-100 text-emerald-700 dark:bg-emerald-900 dark:text-emerald-300">
                            Primary
                          </Badge>
                        )}
                        <Button variant="ghost" size="icon" className="h-8 w-8" onClick={() => openEditMethod(method)}>
                          <Edit2 className="h-4 w-4" />
                        </Button>
                        {!method.isPrimary && (
                          <Button
                            variant="ghost"
                            size="icon"
                            className="h-8 w-8 text-red-500 hover:text-red-600 hover:bg-red-50"
                            onClick={() => handleDeleteMethod(method.id)}
                          >
                            <Trash2 className="h-4 w-4" />
                          </Button>
                        )}
                      </div>
                    </div>
                    {!method.isPrimary && (
                      <Button
                        variant="link"
                        className="mt-2 p-0 h-auto text-sm text-slate-600"
                        onClick={() => handleSetPrimary(method.id)}
                      >
                        Set as Primary
                      </Button>
                    )}
                  </div>
                ))}
              </div>
            )}
          </div>

          <DialogFooter>
            <Button
              onClick={() => {
                resetMethodForm()
                setIsAddMethodDialogOpen(true)
              }}
              className="bg-slate-900 hover:bg-slate-800 dark:bg-white dark:text-slate-900 dark:hover:bg-slate-100"
            >
              <Plus className="h-4 w-4 mr-2" />
              Add Payment Method
            </Button>
          </DialogFooter>
        </DialogContent>
      </Dialog>

      {/* Add Payment Method Dialog */}
      <Dialog open={isAddMethodDialogOpen} onOpenChange={setIsAddMethodDialogOpen}>
        <DialogContent className="max-w-md">
          <DialogHeader>
            <DialogTitle>Add Payment Method</DialogTitle>
          </DialogHeader>

          <div className="space-y-4 py-4">
            <div className="space-y-2">
              <Label>Method Type</Label>
              <Select
                value={methodForm.type}
                onValueChange={(value: "bank" | "upi") => setMethodForm({ ...methodForm, type: value })}
              >
                <SelectTrigger>
                  <SelectValue />
                </SelectTrigger>
                <SelectContent>
                  <SelectItem value="bank">Bank Account</SelectItem>
                  <SelectItem value="upi">UPI</SelectItem>
                </SelectContent>
              </Select>
            </div>

            {methodForm.type === "bank" ? (
              <>
                <div className="space-y-2">
                  <Label htmlFor="bankName">Bank Name</Label>
                  <Input
                    id="bankName"
                    placeholder="e.g., HDFC Bank"
                    value={methodForm.bankName}
                    onChange={(e) => setMethodForm({ ...methodForm, bankName: e.target.value })}
                  />
                </div>
                <div className="space-y-2">
                  <Label htmlFor="accountNumber">Account Number</Label>
                  <Input
                    id="accountNumber"
                    placeholder="Enter account number"
                    value={methodForm.accountNumber}
                    onChange={(e) => setMethodForm({ ...methodForm, accountNumber: e.target.value })}
                  />
                </div>
                <div className="space-y-2">
                  <Label htmlFor="ifscCode">IFSC Code</Label>
                  <Input
                    id="ifscCode"
                    placeholder="e.g., HDFC0001234"
                    value={methodForm.ifscCode}
                    onChange={(e) => setMethodForm({ ...methodForm, ifscCode: e.target.value })}
                  />
                </div>
                <div className="space-y-2">
                  <Label htmlFor="accountHolder">Account Holder Name</Label>
                  <Input
                    id="accountHolder"
                    placeholder="Name as per bank records"
                    value={methodForm.accountHolder}
                    onChange={(e) => setMethodForm({ ...methodForm, accountHolder: e.target.value })}
                  />
                </div>
              </>
            ) : (
              <div className="space-y-2">
                <Label htmlFor="upiId">UPI ID</Label>
                <Input
                  id="upiId"
                  placeholder="e.g., yourname@upi"
                  value={methodForm.upiId}
                  onChange={(e) => setMethodForm({ ...methodForm, upiId: e.target.value })}
                />
              </div>
            )}
          </div>

          <DialogFooter>
            <Button variant="outline" onClick={() => setIsAddMethodDialogOpen(false)}>
              Cancel
            </Button>
            <Button
              onClick={handleAddMethod}
              className="bg-slate-900 hover:bg-slate-800 dark:bg-white dark:text-slate-900 dark:hover:bg-slate-100"
            >
              Add Method
            </Button>
          </DialogFooter>
        </DialogContent>
      </Dialog>

      {/* Edit Payment Method Dialog */}
      <Dialog open={isEditMethodDialogOpen} onOpenChange={setIsEditMethodDialogOpen}>
        <DialogContent className="max-w-md">
          <DialogHeader>
            <DialogTitle>Edit Payment Method</DialogTitle>
          </DialogHeader>

          <div className="space-y-4 py-4">
            <div className="space-y-2">
              <Label>Method Type</Label>
              <Select
                value={methodForm.type}
                onValueChange={(value: "bank" | "upi") => setMethodForm({ ...methodForm, type: value })}
              >
                <SelectTrigger>
                  <SelectValue />
                </SelectTrigger>
                <SelectContent>
                  <SelectItem value="bank">Bank Account</SelectItem>
                  <SelectItem value="upi">UPI</SelectItem>
                </SelectContent>
              </Select>
            </div>

            {methodForm.type === "bank" ? (
              <>
                <div className="space-y-2">
                  <Label htmlFor="editBankName">Bank Name</Label>
                  <Input
                    id="editBankName"
                    placeholder="e.g., HDFC Bank"
                    value={methodForm.bankName}
                    onChange={(e) => setMethodForm({ ...methodForm, bankName: e.target.value })}
                  />
                </div>
                <div className="space-y-2">
                  <Label htmlFor="editAccountNumber">Account Number</Label>
                  <Input
                    id="editAccountNumber"
                    placeholder="Enter account number"
                    value={methodForm.accountNumber}
                    onChange={(e) => setMethodForm({ ...methodForm, accountNumber: e.target.value })}
                  />
                </div>
                <div className="space-y-2">
                  <Label htmlFor="editIfscCode">IFSC Code</Label>
                  <Input
                    id="editIfscCode"
                    placeholder="e.g., HDFC0001234"
                    value={methodForm.ifscCode}
                    onChange={(e) => setMethodForm({ ...methodForm, ifscCode: e.target.value })}
                  />
                </div>
                <div className="space-y-2">
                  <Label htmlFor="editAccountHolder">Account Holder Name</Label>
                  <Input
                    id="editAccountHolder"
                    placeholder="Name as per bank records"
                    value={methodForm.accountHolder}
                    onChange={(e) => setMethodForm({ ...methodForm, accountHolder: e.target.value })}
                  />
                </div>
              </>
            ) : (
              <div className="space-y-2">
                <Label htmlFor="editUpiId">UPI ID</Label>
                <Input
                  id="editUpiId"
                  placeholder="e.g., yourname@upi"
                  value={methodForm.upiId}
                  onChange={(e) => setMethodForm({ ...methodForm, upiId: e.target.value })}
                />
              </div>
            )}
          </div>

          <DialogFooter>
            <Button variant="outline" onClick={() => setIsEditMethodDialogOpen(false)}>
              Cancel
            </Button>
            <Button
              onClick={handleEditMethod}
              className="bg-slate-900 hover:bg-slate-800 dark:bg-white dark:text-slate-900 dark:hover:bg-slate-100"
            >
              Save Changes
            </Button>
          </DialogFooter>
        </DialogContent>
      </Dialog>

      {/* Withdraw Now Dialog */}
      <Dialog open={isWithdrawDialogOpen} onOpenChange={setIsWithdrawDialogOpen}>
        <DialogContent className="max-w-md">
          <DialogHeader>
            <DialogTitle className="flex items-center gap-2">
              <ArrowDownToLine className="h-5 w-5" />
              Withdraw Funds
            </DialogTitle>
            <DialogDescription>Request a withdrawal to your linked account</DialogDescription>
          </DialogHeader>

          <div className="space-y-4 py-4">
            <div className="p-4 rounded-lg bg-slate-100 dark:bg-slate-800">
              <div className="flex items-center justify-between">
                <span className="text-sm text-muted-foreground">Available Balance</span>
                <span className="text-xl font-bold">₹2,85,000</span>
              </div>
            </div>

            <div className="space-y-2">
              <Label htmlFor="withdrawAmount">Amount to Withdraw</Label>
              <div className="relative">
                <span className="absolute left-3 top-1/2 -translate-y-1/2 text-muted-foreground">₹</span>
                <Input
                  id="withdrawAmount"
                  type="number"
                  placeholder="Enter amount"
                  className="pl-8"
                  value={withdrawForm.amount}
                  onChange={(e) => setWithdrawForm({ ...withdrawForm, amount: e.target.value })}
                />
              </div>
              <div className="flex gap-2">
                {["10000", "25000", "50000", "100000"].map((amount) => (
                  <Button
                    key={amount}
                    variant="outline"
                    size="sm"
                    className="flex-1 bg-transparent"
                    onClick={() => setWithdrawForm({ ...withdrawForm, amount })}
                  >
                    ₹{Number(amount).toLocaleString("en-IN")}
                  </Button>
                ))}
              </div>
            </div>

            <div className="space-y-2">
              <Label>Select Payment Method</Label>
              <Select
                value={withdrawForm.methodId}
                onValueChange={(value) => setWithdrawForm({ ...withdrawForm, methodId: value })}
              >
                <SelectTrigger>
                  <SelectValue placeholder="Choose a payment method" />
                </SelectTrigger>
                <SelectContent>
                  {paymentMethods.map((method) => (
                    <SelectItem key={method.id} value={method.id}>
                      <div className="flex items-center gap-2">
                        {method.type === "bank" ? (
                          <>
                            <Building className="h-4 w-4" />
                            {method.bankName} - {method.accountNumber}
                          </>
                        ) : (
                          <>
                            <Wallet className="h-4 w-4" />
                            UPI - {method.upiId}
                          </>
                        )}
                        {method.isPrimary && (
                          <Badge variant="secondary" className="text-xs">
                            Primary
                          </Badge>
                        )}
                      </div>
                    </SelectItem>
                  ))}
                </SelectContent>
              </Select>
            </div>

            {Number(withdrawForm.amount) > 0 && withdrawForm.methodId && (
              <div className="p-4 rounded-lg bg-emerald-50 dark:bg-emerald-950/30 border border-emerald-200 dark:border-emerald-800">
                <div className="flex items-start gap-2">
                  <CheckCircle className="h-5 w-5 text-emerald-600 dark:text-emerald-400 mt-0.5" />
                  <div className="text-sm">
                    <p className="font-medium text-emerald-700 dark:text-emerald-300">Ready to withdraw</p>
                    <p className="text-emerald-600 dark:text-emerald-400">
                      ₹{Number(withdrawForm.amount).toLocaleString("en-IN")} will be transferred within 2-3 business
                      days
                    </p>
                  </div>
                </div>
              </div>
            )}

            <div className="p-3 rounded-lg bg-amber-50 dark:bg-amber-950/30 border border-amber-200 dark:border-amber-800">
              <div className="flex items-start gap-2">
                <AlertCircle className="h-4 w-4 text-amber-600 dark:text-amber-400 mt-0.5" />
                <p className="text-xs text-amber-700 dark:text-amber-300">
                  Minimum withdrawal: ₹1,000. Processing time: 2-3 business days.
                </p>
              </div>
            </div>
          </div>

          <DialogFooter>
            <Button variant="outline" onClick={() => setIsWithdrawDialogOpen(false)}>
              Cancel
            </Button>
            <Button
              onClick={handleWithdrawSubmit}
              disabled={!withdrawForm.amount || !withdrawForm.methodId || Number(withdrawForm.amount) < 1000}
              className="bg-slate-900 hover:bg-slate-800 dark:bg-white dark:text-slate-900 dark:hover:bg-slate-100"
            >
              <ArrowDownToLine className="h-4 w-4 mr-2" />
              Request Withdrawal
            </Button>
          </DialogFooter>
        </DialogContent>
      </Dialog>
    </div>
  )
}
