"use server"

import { cookies } from "next/headers"

const SESSION_COOKIE_NAME = "admin_session"
const RESET_FLOW_COOKIE = "reset_flow_state"

interface SessionData {
  userId: string
  email: string
  role: "admin" | "customer" | "agent" | "hotel"
  name?: string
  loginTime?: string
}

interface ResetFlowState {
  email: string
  otpVerified: boolean
  createdAt: number
  expiresAt: number
}

function generateSessionSignature(data: SessionData): string {
  const payload = `${data.userId}:${data.email}:${data.role}`
  let hash = 0
  for (let i = 0; i < payload.length; i++) {
    const char = payload.charCodeAt(i)
    hash = (hash << 5) - hash + char
    hash = hash & hash
  }
  return Math.abs(hash).toString(36)
}

export async function createSession(data: SessionData): Promise<void> {
  const cookieStore = await cookies()
  const signature = generateSessionSignature(data)
  const sessionData = JSON.stringify({ ...data, sig: signature })

  cookieStore.set(SESSION_COOKIE_NAME, sessionData, {
    httpOnly: true,
    secure: process.env.NODE_ENV === "production",
    sameSite: "strict",
    maxAge: 60 * 60 * 24 * 7,
    path: "/",
  })
}

export async function getSession(): Promise<SessionData | null> {
  const cookieStore = await cookies()
  const sessionCookie = cookieStore.get(SESSION_COOKIE_NAME)

  if (!sessionCookie) {
    return null
  }

  try {
    const parsed = JSON.parse(sessionCookie.value)

    if (!parsed.userId || !parsed.email || !parsed.role) {
      return null
    }

    const expectedSig = generateSessionSignature(parsed)
    if (parsed.sig !== expectedSig) {
      return null
    }

    return parsed as SessionData
  } catch {
    return null
  }
}

export async function destroySession(): Promise<void> {
  const cookieStore = await cookies()
  cookieStore.delete(SESSION_COOKIE_NAME)
}

export async function verifyAdminAuth(): Promise<SessionData | null> {
  const session = await getSession()

  if (!session || session.role !== "admin") {
    return null
  }

  return session
}

export async function createResetFlowState(email: string): Promise<void> {
  const cookieStore = await cookies()
  const state: ResetFlowState = {
    email,
    otpVerified: false,
    createdAt: Date.now(),
    expiresAt: Date.now() + 15 * 60 * 1000,
  }

  cookieStore.set(RESET_FLOW_COOKIE, JSON.stringify(state), {
    httpOnly: true,
    secure: process.env.NODE_ENV === "production",
    sameSite: "strict",
    maxAge: 15 * 60,
    path: "/",
  })
}

export async function getResetFlowState(): Promise<ResetFlowState | null> {
  const cookieStore = await cookies()
  const stateCookie = cookieStore.get(RESET_FLOW_COOKIE)

  if (!stateCookie) {
    return null
  }

  try {
    const state = JSON.parse(stateCookie.value) as ResetFlowState

    if (Date.now() > state.expiresAt) {
      await clearResetFlowState()
      return null
    }

    return state
  } catch {
    return null
  }
}

export async function markOTPVerified(): Promise<boolean> {
  const cookieStore = await cookies()
  const state = await getResetFlowState()

  if (!state) {
    return false
  }

  const updatedState: ResetFlowState = {
    ...state,
    otpVerified: true,
  }

  cookieStore.set(RESET_FLOW_COOKIE, JSON.stringify(updatedState), {
    httpOnly: true,
    secure: process.env.NODE_ENV === "production",
    sameSite: "strict",
    maxAge: 15 * 60,
    path: "/",
  })

  return true
}

export async function clearResetFlowState(): Promise<void> {
  const cookieStore = await cookies()
  cookieStore.delete(RESET_FLOW_COOKIE)
}

export async function verifyResetFlowForOTP(): Promise<{ valid: boolean; email?: string }> {
  const state = await getResetFlowState()

  if (!state || !state.email) {
    return { valid: false }
  }

  return { valid: true, email: state.email }
}

export async function verifyResetFlowForPasswordReset(): Promise<{ valid: boolean; email?: string }> {
  const state = await getResetFlowState()

  if (!state || !state.email || !state.otpVerified) {
    return { valid: false }
  }

  return { valid: true, email: state.email }
}

// Re-export types for use in other files
export type { SessionData, ResetFlowState }
