import { customPackageDataStore } from "./custom-packages"

export type AgentBookingStatus = "booked" | "ongoing" | "completed" | "cancelled"

export interface AgentBooking {
  id: string
  // Agent information
  agentId: string
  agentName: string
  agentEmail: string
  agentPhone: string
  // Customer information
  customerId: string
  customerName: string
  // Package information
  packageId: string
  packageCityName: string
  packageCategory: string
  bookingDate: string
  travelersCount: number
  startDate: string
  endDate: string
  totalDays: number
  bookingAmount: number
  advanceAmount: number
  pendingAmount: number
  agentCommission: number
  status: AgentBookingStatus
  selectedVehicle: string
  createdAt: string
  updatedAt: string
}

class AgentBookingDataStore {
  private bookings: Map<string, AgentBooking> = new Map()
  private idCounter = 0

  constructor() {
    this.initializeSampleData()
  }

  private initializeSampleData() {
    const packages = customPackageDataStore.getActivePackages()

    const agentData = [
      { name: "Rajesh Travel Agency", email: "rajesh@travels.com", phone: "+91 98765 43210" },
      { name: "Golden Tours", email: "info@goldentours.com", phone: "+91 98765 43211" },
      { name: "Divine Yatra", email: "divine@yatra.com", phone: "+91 98765 43212" },
      { name: "Sacred Journeys", email: "sacred@journeys.com", phone: "+91 98765 43213" },
      { name: "Tirupati Travels", email: "tirupati@travels.com", phone: "+91 98765 43214" },
    ]

    const customerNames = [
      "Ramesh Kumar Singh",
      "Sunita Devi Sharma",
      "Anil Kumar Patel",
      "Kavita Singh",
      "Suresh Yadav",
      "Meena Kumari",
      "Rajiv Gupta",
      "Anita Verma",
      "Vinod Mishra",
      "Geeta Rani",
      "Prakash Joshi",
      "Rekha Agarwal",
      "Mohan Lal Gupta",
      "Saroj Kumari",
      "Dinesh Choudhary",
      "Usha Devi",
      "Kishan Singh Rathore",
      "Poonam Sharma",
      "Ravi Prasad",
      "Seema Gupta",
    ]

    const sampleBookings: Array<{
      agentIndex: number
      customerName: string
      packageIndex: number
      travelersCount: number
      bookingDate: string
      startDate: string
      status: AgentBookingStatus
    }> = [
      // Booked (upcoming)
      {
        agentIndex: 0,
        customerName: customerNames[0],
        packageIndex: 0,
        travelersCount: 4,
        bookingDate: "2024-02-01",
        startDate: "2024-03-15",
        status: "booked",
      },
      {
        agentIndex: 1,
        customerName: customerNames[1],
        packageIndex: 1,
        travelersCount: 2,
        bookingDate: "2024-02-03",
        startDate: "2024-03-20",
        status: "booked",
      },
      {
        agentIndex: 2,
        customerName: customerNames[2],
        packageIndex: 2,
        travelersCount: 6,
        bookingDate: "2024-02-05",
        startDate: "2024-03-25",
        status: "booked",
      },
      {
        agentIndex: 0,
        customerName: customerNames[3],
        packageIndex: 0,
        travelersCount: 3,
        bookingDate: "2024-02-06",
        startDate: "2024-04-01",
        status: "booked",
      },
      {
        agentIndex: 3,
        customerName: customerNames[4],
        packageIndex: 1,
        travelersCount: 5,
        bookingDate: "2024-02-08",
        startDate: "2024-04-05",
        status: "booked",
      },
      {
        agentIndex: 1,
        customerName: customerNames[5],
        packageIndex: 2,
        travelersCount: 7,
        bookingDate: "2024-02-09",
        startDate: "2024-04-08",
        status: "booked",
      },

      // Ongoing (in progress)
      {
        agentIndex: 2,
        customerName: customerNames[6],
        packageIndex: 0,
        travelersCount: 2,
        bookingDate: "2024-01-20",
        startDate: "2024-02-08",
        status: "ongoing",
      },
      {
        agentIndex: 4,
        customerName: customerNames[7],
        packageIndex: 1,
        travelersCount: 5,
        bookingDate: "2024-01-22",
        startDate: "2024-02-10",
        status: "ongoing",
      },
      {
        agentIndex: 0,
        customerName: customerNames[8],
        packageIndex: 2,
        travelersCount: 4,
        bookingDate: "2024-01-25",
        startDate: "2024-02-11",
        status: "ongoing",
      },
      {
        agentIndex: 3,
        customerName: customerNames[9],
        packageIndex: 0,
        travelersCount: 6,
        bookingDate: "2024-01-28",
        startDate: "2024-02-12",
        status: "ongoing",
      },

      // Completed (past)
      {
        agentIndex: 1,
        customerName: customerNames[10],
        packageIndex: 0,
        travelersCount: 5,
        bookingDate: "2023-12-01",
        startDate: "2024-01-05",
        status: "completed",
      },
      {
        agentIndex: 2,
        customerName: customerNames[11],
        packageIndex: 1,
        travelersCount: 2,
        bookingDate: "2023-12-05",
        startDate: "2024-01-10",
        status: "completed",
      },
      {
        agentIndex: 4,
        customerName: customerNames[12],
        packageIndex: 2,
        travelersCount: 4,
        bookingDate: "2023-12-08",
        startDate: "2024-01-12",
        status: "completed",
      },
      {
        agentIndex: 0,
        customerName: customerNames[13],
        packageIndex: 0,
        travelersCount: 3,
        bookingDate: "2023-12-10",
        startDate: "2024-01-15",
        status: "completed",
      },
      {
        agentIndex: 3,
        customerName: customerNames[14],
        packageIndex: 1,
        travelersCount: 6,
        bookingDate: "2023-12-12",
        startDate: "2024-01-18",
        status: "completed",
      },
      {
        agentIndex: 1,
        customerName: customerNames[15],
        packageIndex: 2,
        travelersCount: 4,
        bookingDate: "2023-12-15",
        startDate: "2024-01-20",
        status: "completed",
      },

      // Cancelled
      {
        agentIndex: 2,
        customerName: customerNames[16],
        packageIndex: 0,
        travelersCount: 4,
        bookingDate: "2024-01-10",
        startDate: "2024-02-20",
        status: "cancelled",
      },
      {
        agentIndex: 4,
        customerName: customerNames[17],
        packageIndex: 1,
        travelersCount: 3,
        bookingDate: "2024-01-12",
        startDate: "2024-02-25",
        status: "cancelled",
      },
      {
        agentIndex: 0,
        customerName: customerNames[18],
        packageIndex: 2,
        travelersCount: 5,
        bookingDate: "2024-01-15",
        startDate: "2024-03-01",
        status: "cancelled",
      },
    ]

    sampleBookings.forEach((booking) => {
      const pkg = packages[booking.packageIndex % packages.length]
      const agent = agentData[booking.agentIndex % agentData.length]

      if (pkg && agent) {
        this.idCounter++
        const id = `agb_${String(this.idCounter).padStart(4, "0")}`
        const now = new Date().toISOString().split("T")[0]

        // Calculate total days and end date
        const totalDays = pkg.dayPlans.length
        const start = new Date(booking.startDate)
        const end = new Date(start)
        end.setDate(end.getDate() + totalDays - 1)
        const endDate = end.toISOString().split("T")[0]

        // Calculate amounts (using day 1 pricing as base)
        const basePrice = pkg.dayPlans[0]?.pricing.b2cWeekdayPerHead || 1500
        const totalAmount = basePrice * booking.travelersCount * totalDays
        const advanceAmount = Math.round(totalAmount * 0.3) // 30% advance
        const pendingAmount = booking.status === "cancelled" ? 0 : totalAmount - advanceAmount
        const agentCommission = booking.status === "cancelled" ? 0 : Math.round(totalAmount * 0.1) // 10% commission

        // Get vehicle name
        const selectedVehicle = pkg.dayPlans[0]?.vehicles[0]?.vehicleName || "Standard Vehicle"

        this.bookings.set(id, {
          id,
          agentId: `agt_${String(booking.agentIndex + 1).padStart(3, "0")}`,
          agentName: agent.name,
          agentEmail: agent.email,
          agentPhone: agent.phone,
          customerId: `cust_${String(this.idCounter).padStart(3, "0")}`,
          customerName: booking.customerName,
          packageId: pkg.id,
          packageCityName: pkg.cityName,
          packageCategory: pkg.category,
          bookingDate: booking.bookingDate,
          travelersCount: booking.travelersCount,
          startDate: booking.startDate,
          endDate,
          totalDays,
          bookingAmount: booking.status === "cancelled" ? 0 : totalAmount,
          advanceAmount: booking.status === "cancelled" ? 0 : advanceAmount,
          pendingAmount,
          agentCommission,
          status: booking.status,
          selectedVehicle,
          createdAt: now,
          updatedAt: now,
        })
      }
    })
  }

  getAllBookings(): AgentBooking[] {
    return Array.from(this.bookings.values()).sort(
      (a, b) => new Date(b.startDate).getTime() - new Date(a.startDate).getTime(),
    )
  }

  getBookingsByStatus(status: AgentBookingStatus): AgentBooking[] {
    return this.getAllBookings().filter((b) => b.status === status)
  }

  getBookedBookings(): AgentBooking[] {
    return this.getBookingsByStatus("booked")
  }

  getOngoingBookings(): AgentBooking[] {
    return this.getBookingsByStatus("ongoing")
  }

  getCompletedBookings(): AgentBooking[] {
    return this.getBookingsByStatus("completed")
  }

  getCancelledBookings(): AgentBooking[] {
    return this.getBookingsByStatus("cancelled")
  }

  getBookingById(id: string): AgentBooking | undefined {
    return this.bookings.get(id)
  }

  getBookingStats(): {
    booked: number
    ongoing: number
    completed: number
    cancelled: number
    total: number
    bookedAmount: number
    ongoingAmount: number
    completedAmount: number
    totalAmount: number
    totalTravelers: number
    totalCommission: number
  } {
    const all = this.getAllBookings()
    const booked = all.filter((b) => b.status === "booked")
    const ongoing = all.filter((b) => b.status === "ongoing")
    const completed = all.filter((b) => b.status === "completed")
    const cancelled = all.filter((b) => b.status === "cancelled")

    return {
      booked: booked.length,
      ongoing: ongoing.length,
      completed: completed.length,
      cancelled: cancelled.length,
      total: all.length,
      bookedAmount: booked.reduce((sum, b) => sum + b.bookingAmount, 0),
      ongoingAmount: ongoing.reduce((sum, b) => sum + b.bookingAmount, 0),
      completedAmount: completed.reduce((sum, b) => sum + b.bookingAmount, 0),
      totalAmount: all.reduce((sum, b) => sum + b.bookingAmount, 0),
      totalTravelers: all.reduce((sum, b) => sum + b.travelersCount, 0),
      totalCommission: all.reduce((sum, b) => sum + b.agentCommission, 0),
    }
  }
}

export const agentBookingDataStore = new AgentBookingDataStore()
