export type AgentStatus = "pending" | "approved" | "rejected" | "blocked"

export interface AgentBooking {
  id: string
  customerName: string
  packageName: string
  destination: string
  travelDate: string
  commission: number
  status: "completed" | "upcoming" | "cancelled"
}

export interface Agent {
  id: string
  firstName: string
  lastName: string
  email: string
  phone: string
  gstCertificate: string
  msmeCertificate: string
  officeAddress: string
  visitingCard: string
  status: AgentStatus
  totalBookings: number
  totalCommission: number
  joinedAt: string
  lastActive: string
  blockReason?: string
  blockedAt?: string
  rejectionReason?: string
  rejectedAt?: string
  approvedAt?: string
  bookings: AgentBooking[]
}

class AgentDataStore {
  private agents: Map<string, Agent> = new Map()

  constructor() {
    this.initializeSampleData()
  }

  private initializeSampleData() {
    const sampleAgents: Agent[] = [
      {
        id: "agent_001",
        firstName: "Ramesh",
        lastName: "Gupta",
        email: "ramesh.gupta@travelagency.com",
        phone: "+91 9876543220",
        gstCertificate: "GST29ABCDE1234F1Z5",
        msmeCertificate: "UDYAM-UP-00-0012345",
        officeAddress: "123 MG Road, Connaught Place, New Delhi - 110001",
        visitingCard: "/documents/visiting-card-001.pdf",
        status: "approved",
        totalBookings: 45,
        totalCommission: 225000,
        joinedAt: "2023-06-15",
        lastActive: "2024-12-01",
        approvedAt: "2023-06-20",
        bookings: [
          {
            id: "ab_001",
            customerName: "Rajesh Kumar",
            packageName: "Char Dham Yatra",
            destination: "Uttarakhand",
            travelDate: "2024-05-15",
            commission: 3500,
            status: "completed",
          },
          {
            id: "ab_002",
            customerName: "Priya Sharma",
            packageName: "Varanasi Tour",
            destination: "Varanasi",
            travelDate: "2024-08-20",
            commission: 1500,
            status: "completed",
          },
        ],
      },
      {
        id: "agent_002",
        firstName: "Sunita",
        lastName: "Desai",
        email: "sunita.desai@yatratours.com",
        phone: "+91 9876543221",
        gstCertificate: "GST27FGHIJ5678K2Z6",
        msmeCertificate: "UDYAM-MH-00-0067890",
        officeAddress: "456 FC Road, Shivaji Nagar, Pune - 411005",
        visitingCard: "/documents/visiting-card-002.pdf",
        status: "approved",
        totalBookings: 32,
        totalCommission: 160000,
        joinedAt: "2023-09-10",
        lastActive: "2024-11-28",
        approvedAt: "2023-09-15",
        bookings: [],
      },
      {
        id: "agent_003",
        firstName: "Anil",
        lastName: "Mehta",
        email: "anil.mehta@pilgrimtravels.com",
        phone: "+91 9876543222",
        gstCertificate: "GST24KLMNO9012P3Z7",
        msmeCertificate: "UDYAM-GJ-00-0034567",
        officeAddress: "789 CG Road, Navrangpura, Ahmedabad - 380009",
        visitingCard: "/documents/visiting-card-003.pdf",
        status: "pending",
        totalBookings: 0,
        totalCommission: 0,
        joinedAt: "2024-11-25",
        lastActive: "2024-11-25",
        bookings: [],
      },
      {
        id: "agent_004",
        firstName: "Kavitha",
        lastName: "Nair",
        email: "kavitha.nair@keralatours.com",
        phone: "+91 9876543223",
        gstCertificate: "GST32QRSTU3456V4Z8",
        msmeCertificate: "UDYAM-KL-00-0078901",
        officeAddress: "321 MG Road, Ernakulam, Kochi - 682016",
        visitingCard: "/documents/visiting-card-004.pdf",
        status: "pending",
        totalBookings: 0,
        totalCommission: 0,
        joinedAt: "2024-12-01",
        lastActive: "2024-12-01",
        bookings: [],
      },
      {
        id: "agent_005",
        firstName: "Vikram",
        lastName: "Joshi",
        email: "vikram.joshi@tirthyatra.com",
        phone: "+91 9876543224",
        gstCertificate: "GST33VWXYZ7890A5Z9",
        msmeCertificate: "UDYAM-TN-00-0023456",
        officeAddress: "654 Anna Salai, T Nagar, Chennai - 600017",
        visitingCard: "/documents/visiting-card-005.pdf",
        status: "rejected",
        totalBookings: 0,
        totalCommission: 0,
        joinedAt: "2024-10-15",
        lastActive: "2024-10-15",
        rejectionReason: "Invalid GST certificate provided. Document verification failed.",
        rejectedAt: "2024-10-20",
        bookings: [],
      },
      {
        id: "agent_006",
        firstName: "Deepa",
        lastName: "Rao",
        email: "deepa.rao@spiritualjourneys.com",
        phone: "+91 9876543225",
        gstCertificate: "GST29BCDEF2345G6Z0",
        msmeCertificate: "UDYAM-KA-00-0045678",
        officeAddress: "987 Brigade Road, Bangalore - 560001",
        visitingCard: "/documents/visiting-card-006.pdf",
        status: "blocked",
        totalBookings: 12,
        totalCommission: 60000,
        joinedAt: "2024-01-20",
        lastActive: "2024-09-10",
        approvedAt: "2024-01-25",
        blockReason: "Multiple customer complaints regarding misleading package information.",
        blockedAt: "2024-09-15",
        bookings: [],
      },
      {
        id: "agent_007",
        firstName: "Suresh",
        lastName: "Pillai",
        email: "suresh.pillai@templetrips.com",
        phone: "+91 9876543226",
        gstCertificate: "GST32CDEFG3456H7Z1",
        msmeCertificate: "UDYAM-KL-00-0056789",
        officeAddress: "147 Beach Road, Trivandrum - 695001",
        visitingCard: "/documents/visiting-card-007.pdf",
        status: "approved",
        totalBookings: 28,
        totalCommission: 140000,
        joinedAt: "2023-12-05",
        lastActive: "2024-11-30",
        approvedAt: "2023-12-10",
        bookings: [],
      },
      {
        id: "agent_008",
        firstName: "Anjali",
        lastName: "Sharma",
        email: "anjali.sharma@divinetours.com",
        phone: "+91 9876543227",
        gstCertificate: "GST06EFGHI4567I8Z2",
        msmeCertificate: "UDYAM-HR-00-0067890",
        officeAddress: "258 Sector 17, Chandigarh - 160017",
        visitingCard: "/documents/visiting-card-008.pdf",
        status: "pending",
        totalBookings: 0,
        totalCommission: 0,
        joinedAt: "2024-11-28",
        lastActive: "2024-11-28",
        bookings: [],
      },
      {
        id: "agent_009",
        firstName: "Manoj",
        lastName: "Verma",
        email: "manoj.verma@sacredpaths.com",
        phone: "+91 9876543228",
        gstCertificate: "GST09GHIJK5678J9Z3",
        msmeCertificate: "UDYAM-UP-00-0078901",
        officeAddress: "369 Hazratganj, Lucknow - 226001",
        visitingCard: "/documents/visiting-card-009.pdf",
        status: "rejected",
        totalBookings: 0,
        totalCommission: 0,
        joinedAt: "2024-09-05",
        lastActive: "2024-09-05",
        rejectionReason: "Office address verification failed. Address does not exist.",
        rejectedAt: "2024-09-12",
        bookings: [],
      },
      {
        id: "agent_010",
        firstName: "Priyanka",
        lastName: "Das",
        email: "priyanka.das@holytravel.com",
        phone: "+91 9876543229",
        gstCertificate: "GST19HIJKL6789K0Z4",
        msmeCertificate: "UDYAM-WB-00-0089012",
        officeAddress: "741 Park Street, Kolkata - 700016",
        visitingCard: "/documents/visiting-card-010.pdf",
        status: "blocked",
        totalBookings: 8,
        totalCommission: 40000,
        joinedAt: "2024-03-15",
        lastActive: "2024-08-20",
        approvedAt: "2024-03-20",
        blockReason: "Fraudulent booking practices detected. Under investigation.",
        blockedAt: "2024-08-25",
        bookings: [],
      },
    ]

    sampleAgents.forEach((agent) => {
      this.agents.set(agent.id, agent)
    })
  }

  getAllAgents(): Agent[] {
    return Array.from(this.agents.values())
  }

  getPendingAgents(): Agent[] {
    return this.getAllAgents().filter((a) => a.status === "pending")
  }

  getApprovedAgents(): Agent[] {
    return this.getAllAgents().filter((a) => a.status === "approved")
  }

  getRejectedAgents(): Agent[] {
    return this.getAllAgents().filter((a) => a.status === "rejected")
  }

  getBlockedAgents(): Agent[] {
    return this.getAllAgents().filter((a) => a.status === "blocked")
  }

  getAgentById(id: string): Agent | undefined {
    return this.agents.get(id)
  }

  approveAgent(id: string): Agent | undefined {
    const agent = this.agents.get(id)
    if (!agent) return undefined

    const updatedAgent: Agent = {
      ...agent,
      status: "approved",
      approvedAt: new Date().toISOString().split("T")[0],
      rejectionReason: undefined,
      rejectedAt: undefined,
    }
    this.agents.set(id, updatedAgent)
    return updatedAgent
  }

  rejectAgent(id: string, reason: string): Agent | undefined {
    const agent = this.agents.get(id)
    if (!agent) return undefined

    const updatedAgent: Agent = {
      ...agent,
      status: "rejected",
      rejectionReason: reason,
      rejectedAt: new Date().toISOString().split("T")[0],
    }
    this.agents.set(id, updatedAgent)
    return updatedAgent
  }

  blockAgent(id: string, reason: string): Agent | undefined {
    const agent = this.agents.get(id)
    if (!agent) return undefined

    const updatedAgent: Agent = {
      ...agent,
      status: "blocked",
      blockReason: reason,
      blockedAt: new Date().toISOString().split("T")[0],
    }
    this.agents.set(id, updatedAgent)
    return updatedAgent
  }

  unblockAgent(id: string): Agent | undefined {
    const agent = this.agents.get(id)
    if (!agent) return undefined

    const updatedAgent: Agent = {
      ...agent,
      status: "approved",
      blockReason: undefined,
      blockedAt: undefined,
    }
    this.agents.set(id, updatedAgent)
    return updatedAgent
  }

  restoreRejectedAgent(id: string): Agent | undefined {
    const agent = this.agents.get(id)
    if (!agent) return undefined

    const updatedAgent: Agent = {
      ...agent,
      status: "pending",
      rejectionReason: undefined,
      rejectedAt: undefined,
    }
    this.agents.set(id, updatedAgent)
    return updatedAgent
  }
}

export const agentDataStore = new AgentDataStore()
