import { fixPackageDataStore } from "./fix-packages"

export type FixPackageBookingStatus = "booked" | "ongoing" | "completed" | "cancelled"

export interface FixPackageBooking {
  id: string
  customerId: string
  customerName: string
  packageId: string
  packageName: string
  bookingDate: string
  travelersCount: number
  startDate: string
  endDate: string
  totalDays: number
  bookingAmount: number
  advanceAmount: number
  pendingAmount: number
  status: FixPackageBookingStatus
  location: string
  vehicleSelected: string
  createdAt: string
  updatedAt: string
}

// In-memory fix package booking data store
class FixPackageBookingDataStore {
  private bookings: Map<string, FixPackageBooking> = new Map()
  private idCounter = 0

  constructor() {
    this.initializeSampleData()
  }

  private initializeSampleData() {
    const packages = fixPackageDataStore.getActivePackages()

    const customerNames = [
      "Ramesh Kumar Singh",
      "Sunita Devi Sharma",
      "Anil Kumar Patel",
      "Kavita Singh",
      "Suresh Yadav",
      "Meena Kumari",
      "Rajiv Gupta",
      "Anita Verma",
      "Vinod Mishra",
      "Geeta Rani",
      "Prakash Joshi",
      "Rekha Agarwal",
      "Mohan Lal Gupta",
      "Saroj Kumari",
      "Dinesh Choudhary",
      "Usha Devi",
      "Kishan Singh Rathore",
      "Poonam Sharma",
      "Ravi Prasad",
      "Seema Gupta",
      "Ashok Kumar",
      "Neelam Devi",
      "Sanjay Verma",
      "Shanti Devi",
      "Deepak Rao",
      "Preeti Singh",
      "Manoj Kumar",
      "Anjali Patel",
    ]

    const sampleBookings: Array<{
      customerName: string
      packageIndex: number
      travelersCount: number
      bookingDate: string
      startDate: string
      status: FixPackageBookingStatus
    }> = [
      // Booked (upcoming)
      {
        customerName: customerNames[0],
        packageIndex: 0,
        travelersCount: 4,
        bookingDate: "2024-02-01",
        startDate: "2024-03-15",
        status: "booked",
      },
      {
        customerName: customerNames[1],
        packageIndex: 1,
        travelersCount: 2,
        bookingDate: "2024-02-03",
        startDate: "2024-03-20",
        status: "booked",
      },
      {
        customerName: customerNames[2],
        packageIndex: 2,
        travelersCount: 6,
        bookingDate: "2024-02-05",
        startDate: "2024-03-25",
        status: "booked",
      },
      {
        customerName: customerNames[3],
        packageIndex: 3,
        travelersCount: 3,
        bookingDate: "2024-02-06",
        startDate: "2024-04-01",
        status: "booked",
      },
      {
        customerName: customerNames[4],
        packageIndex: 6,
        travelersCount: 5,
        bookingDate: "2024-02-08",
        startDate: "2024-04-05",
        status: "booked",
      },
      {
        customerName: customerNames[5],
        packageIndex: 7,
        travelersCount: 4,
        bookingDate: "2024-02-10",
        startDate: "2024-04-10",
        status: "booked",
      },
      {
        customerName: customerNames[6],
        packageIndex: 0,
        travelersCount: 7,
        bookingDate: "2024-02-11",
        startDate: "2024-04-15",
        status: "booked",
      },

      // Ongoing (in progress)
      {
        customerName: customerNames[7],
        packageIndex: 1,
        travelersCount: 2,
        bookingDate: "2024-01-20",
        startDate: "2024-02-08",
        status: "ongoing",
      },
      {
        customerName: customerNames[8],
        packageIndex: 2,
        travelersCount: 5,
        bookingDate: "2024-01-22",
        startDate: "2024-02-10",
        status: "ongoing",
      },
      {
        customerName: customerNames[9],
        packageIndex: 3,
        travelersCount: 4,
        bookingDate: "2024-01-25",
        startDate: "2024-02-11",
        status: "ongoing",
      },
      {
        customerName: customerNames[10],
        packageIndex: 6,
        travelersCount: 6,
        bookingDate: "2024-01-28",
        startDate: "2024-02-12",
        status: "ongoing",
      },
      {
        customerName: customerNames[11],
        packageIndex: 0,
        travelersCount: 8,
        bookingDate: "2024-01-30",
        startDate: "2024-02-13",
        status: "ongoing",
      },

      // Completed (past)
      {
        customerName: customerNames[12],
        packageIndex: 0,
        travelersCount: 5,
        bookingDate: "2023-12-01",
        startDate: "2024-01-05",
        status: "completed",
      },
      {
        customerName: customerNames[13],
        packageIndex: 1,
        travelersCount: 2,
        bookingDate: "2023-12-05",
        startDate: "2024-01-10",
        status: "completed",
      },
      {
        customerName: customerNames[14],
        packageIndex: 2,
        travelersCount: 4,
        bookingDate: "2023-12-08",
        startDate: "2024-01-12",
        status: "completed",
      },
      {
        customerName: customerNames[15],
        packageIndex: 3,
        travelersCount: 3,
        bookingDate: "2023-12-10",
        startDate: "2024-01-15",
        status: "completed",
      },
      {
        customerName: customerNames[16],
        packageIndex: 6,
        travelersCount: 6,
        bookingDate: "2023-12-12",
        startDate: "2024-01-18",
        status: "completed",
      },
      {
        customerName: customerNames[17],
        packageIndex: 7,
        travelersCount: 4,
        bookingDate: "2023-12-15",
        startDate: "2024-01-20",
        status: "completed",
      },
      {
        customerName: customerNames[18],
        packageIndex: 0,
        travelersCount: 7,
        bookingDate: "2023-12-18",
        startDate: "2024-01-22",
        status: "completed",
      },
      {
        customerName: customerNames[19],
        packageIndex: 1,
        travelersCount: 2,
        bookingDate: "2023-12-20",
        startDate: "2024-01-25",
        status: "completed",
      },
      {
        customerName: customerNames[20],
        packageIndex: 2,
        travelersCount: 5,
        bookingDate: "2023-12-22",
        startDate: "2024-01-28",
        status: "completed",
      },

      // Cancelled
      {
        customerName: customerNames[21],
        packageIndex: 0,
        travelersCount: 4,
        bookingDate: "2024-01-10",
        startDate: "2024-02-20",
        status: "cancelled",
      },
      {
        customerName: customerNames[22],
        packageIndex: 3,
        travelersCount: 3,
        bookingDate: "2024-01-12",
        startDate: "2024-02-25",
        status: "cancelled",
      },
      {
        customerName: customerNames[23],
        packageIndex: 6,
        travelersCount: 5,
        bookingDate: "2024-01-15",
        startDate: "2024-03-01",
        status: "cancelled",
      },
      {
        customerName: customerNames[24],
        packageIndex: 1,
        travelersCount: 2,
        bookingDate: "2024-01-18",
        startDate: "2024-03-05",
        status: "cancelled",
      },
    ]

    sampleBookings.forEach((booking) => {
      const pkg = packages[booking.packageIndex % packages.length]

      if (pkg) {
        this.idCounter++
        const id = `fpb_${String(this.idCounter).padStart(4, "0")}`
        const now = new Date().toISOString().split("T")[0]

        // Calculate end date based on package duration
        const start = new Date(booking.startDate)
        const end = new Date(start)
        end.setDate(end.getDate() + pkg.totalDays - 1)
        const endDate = end.toISOString().split("T")[0]

        // Calculate amounts
        const totalAmount = pkg.discountedPrice * booking.travelersCount
        const advanceAmount = Math.round(totalAmount * 0.3) // 30% advance
        const pendingAmount = booking.status === "cancelled" ? 0 : totalAmount - advanceAmount

        // Get vehicle name if available
        const vehicleSelected = pkg.vehicles.length > 0 ? pkg.vehicles[0].vehicleName : "Standard Vehicle Package"

        this.bookings.set(id, {
          id,
          customerId: `cust_${String(this.idCounter).padStart(3, "0")}`,
          customerName: booking.customerName,
          packageId: pkg.id,
          packageName: pkg.name,
          bookingDate: booking.bookingDate,
          travelersCount: booking.travelersCount,
          startDate: booking.startDate,
          endDate,
          totalDays: pkg.totalDays,
          bookingAmount: booking.status === "cancelled" ? 0 : totalAmount,
          advanceAmount: booking.status === "cancelled" ? 0 : advanceAmount,
          pendingAmount,
          status: booking.status,
          location: pkg.locationDetails.split("-")[0].trim(),
          vehicleSelected,
          createdAt: now,
          updatedAt: now,
        })
      }
    })
  }

  getAllBookings(): FixPackageBooking[] {
    return Array.from(this.bookings.values()).sort(
      (a, b) => new Date(b.startDate).getTime() - new Date(a.startDate).getTime(),
    )
  }

  getBookingsByStatus(status: FixPackageBookingStatus): FixPackageBooking[] {
    return this.getAllBookings().filter((b) => b.status === status)
  }

  getBookedBookings(): FixPackageBooking[] {
    return this.getBookingsByStatus("booked")
  }

  getOngoingBookings(): FixPackageBooking[] {
    return this.getBookingsByStatus("ongoing")
  }

  getCompletedBookings(): FixPackageBooking[] {
    return this.getBookingsByStatus("completed")
  }

  getCancelledBookings(): FixPackageBooking[] {
    return this.getBookingsByStatus("cancelled")
  }

  getBookingById(id: string): FixPackageBooking | undefined {
    return this.bookings.get(id)
  }

  getBookingStats(): {
    booked: number
    ongoing: number
    completed: number
    cancelled: number
    total: number
    bookedAmount: number
    ongoingAmount: number
    completedAmount: number
    totalAmount: number
    totalTravelers: number
  } {
    const all = this.getAllBookings()
    const booked = all.filter((b) => b.status === "booked")
    const ongoing = all.filter((b) => b.status === "ongoing")
    const completed = all.filter((b) => b.status === "completed")
    const cancelled = all.filter((b) => b.status === "cancelled")

    return {
      booked: booked.length,
      ongoing: ongoing.length,
      completed: completed.length,
      cancelled: cancelled.length,
      total: all.length,
      bookedAmount: booked.reduce((sum, b) => sum + b.bookingAmount, 0),
      ongoingAmount: ongoing.reduce((sum, b) => sum + b.bookingAmount, 0),
      completedAmount: completed.reduce((sum, b) => sum + b.bookingAmount, 0),
      totalAmount: all.reduce((sum, b) => sum + b.bookingAmount, 0),
      totalTravelers: all.reduce((sum, b) => sum + b.travelersCount, 0),
    }
  }
}

export const fixPackageBookingDataStore = new FixPackageBookingDataStore()
