export type FixPackageStatus = "active" | "inactive"

export interface PackageVehicle {
  id: string
  photo: string
  vehicleName: string
  perDayPrice: number
  passengerLimit: number
}

export interface FixPackage {
  id: string
  name: string
  locationDetails: string
  itineraryDetails: string
  otherDetails: string
  originalPrice: number
  discountedPrice: number
  totalDays: number
  vehicles: PackageVehicle[]
  status: FixPackageStatus
  createdAt: string
  updatedAt: string
}

// In-memory fix package data store
class FixPackageDataStore {
  private packages: Map<string, FixPackage> = new Map()
  private idCounter = 0

  constructor() {
    this.initializeSampleData()
  }

  private initializeSampleData() {
    const samplePackages: Omit<FixPackage, "id">[] = [
      {
        name: "Char Dham Yatra",
        locationDetails: "Yamunotri, Gangotri, Kedarnath, Badrinath - Uttarakhand",
        itineraryDetails:
          "Day 1: Haridwar arrival and transfer to Barkot. Day 2: Yamunotri temple visit. Day 3: Gangotri temple visit. Day 4-5: Kedarnath darshan. Day 6-7: Badrinath darshan. Day 8: Return to Haridwar.",
        otherDetails:
          "Package includes: Accommodation, meals (breakfast and dinner), transportation, and guide services. Excludes: Personal expenses, helicopter services, and pony rides.",
        originalPrice: 45000,
        discountedPrice: 38000,
        totalDays: 8,
        vehicles: [
          {
            id: "v1",
            photo: "/placeholder.svg?height=100&width=150",
            vehicleName: "Luxury SUV",
            perDayPrice: 5000,
            passengerLimit: 7,
          },
          {
            id: "v2",
            photo: "/placeholder.svg?height=100&width=150",
            vehicleName: "Tempo Traveller",
            perDayPrice: 6500,
            passengerLimit: 12,
          },
        ],
        status: "active",
        createdAt: "2024-01-10",
        updatedAt: "2024-01-10",
      },
      {
        name: "Golden Triangle Spiritual Tour",
        locationDetails: "Delhi, Agra, Mathura, Vrindavan, Jaipur",
        itineraryDetails:
          "Day 1: Delhi - Akshardham, Lotus Temple, ISKCON Temple. Day 2: Agra - Taj Mahal, then drive to Mathura. Day 3: Mathura and Vrindavan temple tour. Day 4: Jaipur - Govind Dev Ji Temple, City Palace. Day 5: Return journey.",
        otherDetails:
          "Includes: 4-star hotel accommodation, all meals, AC transportation, English-speaking guide, and entrance fees. Excludes: Camera fees, tips, and personal expenses.",
        originalPrice: 32000,
        discountedPrice: 28000,
        totalDays: 5,
        vehicles: [
          {
            id: "v1",
            photo: "/placeholder.svg?height=100&width=150",
            vehicleName: "Sedan",
            perDayPrice: 3000,
            passengerLimit: 4,
          },
        ],
        status: "active",
        createdAt: "2024-01-12",
        updatedAt: "2024-02-15",
      },
      {
        name: "South India Temple Circuit",
        locationDetails: "Chennai, Mahabalipuram, Kanchipuram, Tirupati, Madurai, Rameswaram",
        itineraryDetails:
          "Day 1: Chennai arrival, Kapaleeshwarar Temple. Day 2: Mahabalipuram shore temples, Kanchipuram Kamakshi Temple. Day 3: Tirupati Balaji darshan. Day 4: Madurai Meenakshi Temple. Day 5-6: Rameswaram temple complex. Day 7: Departure.",
        otherDetails:
          "Package includes: Hotel stays, breakfast and dinner, AC coach, VIP darshan tickets where available. Excludes: Lunch, special puja expenses, and donations.",
        originalPrice: 42000,
        discountedPrice: 36000,
        totalDays: 7,
        vehicles: [],
        status: "active",
        createdAt: "2024-01-15",
        updatedAt: "2024-01-15",
      },
      {
        name: "Varanasi - Prayagraj Spiritual Journey",
        locationDetails: "Varanasi, Sarnath, Prayagraj (Allahabad)",
        itineraryDetails:
          "Day 1: Varanasi arrival, evening Ganga Aarti. Day 2: Morning boat ride, Kashi Vishwanath Temple, Sarnath visit. Day 3: Travel to Prayagraj, Triveni Sangam, temples tour. Day 4: Return to Varanasi for departure.",
        otherDetails:
          "Includes: Heritage hotel accommodation, all meals, boat rides, temple guide, and transportation. Excludes: Special pujas, priest dakshina, and personal shopping.",
        originalPrice: 25000,
        discountedPrice: 21000,
        totalDays: 4,
        vehicles: [],
        status: "active",
        createdAt: "2024-01-18",
        updatedAt: "2024-01-18",
      },
      {
        name: "Dwarka - Somnath Coastal Pilgrimage",
        locationDetails: "Dwarka, Bet Dwarka, Somnath, Diu - Gujarat",
        itineraryDetails:
          "Day 1: Dwarka arrival, Dwarkadhish Temple evening aarti. Day 2: Bet Dwarka boat trip, Nageshwar Jyotirlinga. Day 3: Drive to Somnath, evening temple visit. Day 4: Somnath morning aarti, Diu sightseeing. Day 5: Return journey.",
        otherDetails:
          "Package includes: Sea-view hotel accommodation, all vegetarian meals, boat rides, AC transportation, and guided tours. Excludes: Camera fees and personal expenses.",
        originalPrice: 30000,
        discountedPrice: 26000,
        totalDays: 5,
        vehicles: [],
        status: "inactive",
        createdAt: "2024-01-20",
        updatedAt: "2024-03-01",
      },
      {
        name: "Amarnath Yatra Package",
        locationDetails: "Srinagar, Pahalgam, Amarnath Cave, Kashmir",
        itineraryDetails:
          "Day 1: Srinagar arrival, local sightseeing. Day 2: Transfer to Pahalgam. Day 3-4: Amarnath cave trek/helicopter ride (weather permitting). Day 5: Return to Srinagar. Day 6: Departure.",
        otherDetails:
          "Includes: Deluxe hotel/camp accommodation, meals, permits, basic medical kit, and ponies/helicopter (as per option chosen). Excludes: Personal trekking gear, insurance, and emergency evacuation.",
        originalPrice: 55000,
        discountedPrice: 48000,
        totalDays: 6,
        vehicles: [],
        status: "inactive",
        createdAt: "2024-02-01",
        updatedAt: "2024-02-01",
      },
      {
        name: "Kerala Temple & Backwater Tour",
        locationDetails: "Guruvayur, Sabarimala, Trivandrum, Alleppey",
        itineraryDetails:
          "Day 1: Cochin to Guruvayur temple. Day 2-3: Sabarimala pilgrimage (seasonal). Day 4: Padmanabhaswamy Temple, Trivandrum. Day 5: Alleppey backwater houseboat. Day 6: Return journey.",
        otherDetails:
          "Package includes: Houseboat stay, hotel accommodation, all meals, temple guide, and transportation. Excludes: Special entry darshan fees and aravana prasadam purchases.",
        originalPrice: 38000,
        discountedPrice: 33000,
        totalDays: 6,
        vehicles: [],
        status: "active",
        createdAt: "2024-02-05",
        updatedAt: "2024-02-05",
      },
      {
        name: "Shirdi - Nashik - Trimbakeshwar",
        locationDetails: "Shirdi, Nashik, Trimbakeshwar - Maharashtra",
        itineraryDetails:
          "Day 1: Shirdi Sai Baba temple darshan. Day 2: Shani Shingnapur, travel to Nashik. Day 3: Trimbakeshwar Jyotirlinga, Panchavati temples. Day 4: Departure from Nashik.",
        otherDetails:
          "Includes: Hotel stays, breakfast and dinner, AC car, and temple guide services. Excludes: Lunch, special darshan tickets, and donations.",
        originalPrice: 18000,
        discountedPrice: 15000,
        totalDays: 4,
        vehicles: [],
        status: "active",
        createdAt: "2024-02-08",
        updatedAt: "2024-02-08",
      },
    ]

    samplePackages.forEach((pkg) => {
      this.addPackage(
        pkg.name,
        pkg.locationDetails,
        pkg.itineraryDetails,
        pkg.otherDetails,
        pkg.originalPrice,
        pkg.discountedPrice,
        pkg.totalDays,
        pkg.vehicles,
        pkg.status,
      )
    })
  }

  // Get all packages
  getAllPackages(): FixPackage[] {
    return Array.from(this.packages.values()).sort((a, b) => a.name.localeCompare(b.name))
  }

  // Get active packages
  getActivePackages(): FixPackage[] {
    return this.getAllPackages().filter((p) => p.status === "active")
  }

  // Get inactive packages
  getInactivePackages(): FixPackage[] {
    return this.getAllPackages().filter((p) => p.status === "inactive")
  }

  // Get package by ID
  getPackageById(id: string): FixPackage | undefined {
    return this.packages.get(id)
  }

  // Get package by name
  getPackageByName(name: string): FixPackage | undefined {
    return Array.from(this.packages.values()).find((p) => p.name.toLowerCase() === name.toLowerCase())
  }

  addPackage(
    name: string,
    locationDetails: string,
    itineraryDetails: string,
    otherDetails: string,
    originalPrice: number,
    discountedPrice: number,
    totalDays: number,
    vehicles: PackageVehicle[] = [],
    status: FixPackageStatus = "active",
  ): FixPackage | null {
    // Check for duplicate
    if (this.getPackageByName(name)) {
      return null
    }

    this.idCounter++
    const id = `pkg_${String(this.idCounter).padStart(3, "0")}`
    const now = new Date().toISOString().split("T")[0]

    const newPackage: FixPackage = {
      id,
      name: name.trim(),
      locationDetails: locationDetails.trim(),
      itineraryDetails: itineraryDetails.trim(),
      otherDetails: otherDetails.trim(),
      originalPrice,
      discountedPrice,
      totalDays,
      vehicles,
      status,
      createdAt: now,
      updatedAt: now,
    }

    this.packages.set(id, newPackage)
    return newPackage
  }

  updatePackage(
    id: string,
    name: string,
    locationDetails: string,
    itineraryDetails: string,
    otherDetails: string,
    originalPrice: number,
    discountedPrice: number,
    totalDays: number,
    vehicles: PackageVehicle[],
  ): FixPackage | null {
    const existingPackage = this.packages.get(id)
    if (!existingPackage) return null

    // Check for duplicate name (excluding current package)
    const duplicate = this.getPackageByName(name)
    if (duplicate && duplicate.id !== id) {
      return null
    }

    const updatedPackage: FixPackage = {
      ...existingPackage,
      name: name.trim(),
      locationDetails: locationDetails.trim(),
      itineraryDetails: itineraryDetails.trim(),
      otherDetails: otherDetails.trim(),
      originalPrice,
      discountedPrice,
      totalDays,
      vehicles,
      updatedAt: new Date().toISOString().split("T")[0],
    }

    this.packages.set(id, updatedPackage)
    return updatedPackage
  }

  // Toggle package status
  togglePackageStatus(id: string): FixPackage | null {
    const pkg = this.packages.get(id)
    if (!pkg) return null

    const updatedPackage: FixPackage = {
      ...pkg,
      status: pkg.status === "active" ? "inactive" : "active",
      updatedAt: new Date().toISOString().split("T")[0],
    }

    this.packages.set(id, updatedPackage)
    return updatedPackage
  }

  // Delete package
  deletePackage(id: string): boolean {
    return this.packages.delete(id)
  }

  // Get package statistics
  getPackageStats() {
    const all = this.getAllPackages()
    const active = all.filter((p) => p.status === "active")
    const inactive = all.filter((p) => p.status === "inactive")

    return {
      total: all.length,
      active: active.length,
      inactive: inactive.length,
      totalRevenue: all.reduce((sum, p) => sum + p.discountedPrice, 0),
      averagePrice: all.length > 0 ? Math.round(all.reduce((sum, p) => sum + p.discountedPrice, 0) / all.length) : 0,
      averageDuration: all.length > 0 ? Math.round(all.reduce((sum, p) => sum + p.totalDays, 0) / all.length) : 0,
    }
  }
}

export const fixPackageDataStore = new FixPackageDataStore()
