import { locationDataStore } from "./locations"

export type HotelPartnerStatus = "active" | "inactive"

export interface HotelDocument {
  gstNumber: string
  panNumber: string
  registrationNumber: string
  licenseNumber: string
  otherDocuments: string[]
}

export interface HotelPartner {
  id: string
  hotelName: string
  ownerName: string
  stateId: string
  stateName: string
  cityId: string
  cityName: string
  email: string
  phone: string
  password: string
  address: string
  documents: HotelDocument
  since: string
  status: HotelPartnerStatus
  createdAt: string
  updatedAt: string
}

// In-memory hotel partner data store
class HotelPartnerDataStore {
  private partners: Map<string, HotelPartner> = new Map()
  private idCounter = 0

  constructor() {
    this.initializeSampleData()
  }

  private initializeSampleData() {
    const states = locationDataStore.getAllStates()
    const samplePartners = [
      {
        hotelName: "Ganga View Resort",
        ownerName: "Rajesh Kumar",
        stateName: "Uttar Pradesh",
        cityName: "Varanasi",
        email: "ganga.view@email.com",
        phone: "+91 9876543210",
        password: "Hotel@123",
        address: "Assi Ghat, Varanasi, UP 221005",
        gstNumber: "09AABCU9603R1ZM",
        panNumber: "AABCU9603R",
        registrationNumber: "UPHOTEL/2020/12345",
        licenseNumber: "LIC/VAR/2020/001",
        since: "2020-03-15",
      },
      {
        hotelName: "Himalayan Heights",
        ownerName: "Priya Sharma",
        stateName: "Uttarakhand",
        cityName: "Rishikesh",
        email: "himalayan.heights@email.com",
        phone: "+91 9876543211",
        password: "Hotel@456",
        address: "Laxman Jhula Road, Rishikesh, UK 249302",
        gstNumber: "05BBDCS1234A1Z5",
        panNumber: "BBDCS1234A",
        registrationNumber: "UKHOTEL/2019/54321",
        licenseNumber: "LIC/RSH/2019/002",
        since: "2019-06-20",
      },
      {
        hotelName: "Royal Heritage Inn",
        ownerName: "Vikram Singh",
        stateName: "Rajasthan",
        cityName: "Jaipur",
        email: "royal.heritage@email.com",
        phone: "+91 9876543212",
        password: "Hotel@789",
        address: "Amber Road, Jaipur, RJ 302001",
        gstNumber: "08CCDCS5678B2Z6",
        panNumber: "CCDCS5678B",
        registrationNumber: "RJHOTEL/2021/67890",
        licenseNumber: "LIC/JAI/2021/003",
        since: "2021-01-10",
      },
      {
        hotelName: "Dwarka Darshan Hotel",
        ownerName: "Meera Patel",
        stateName: "Gujarat",
        cityName: "Dwarka",
        email: "dwarka.darshan@email.com",
        phone: "+91 9876543213",
        password: "Hotel@101",
        address: "Temple Road, Dwarka, GJ 361335",
        gstNumber: "24DDECS9012C3Z7",
        panNumber: "DDECS9012C",
        registrationNumber: "GJHOTEL/2018/11223",
        licenseNumber: "LIC/DWK/2018/004",
        since: "2018-11-25",
      },
      {
        hotelName: "Shirdi Sai Residency",
        ownerName: "Anil Deshmukh",
        stateName: "Maharashtra",
        cityName: "Shirdi",
        email: "sai.residency@email.com",
        phone: "+91 9876543214",
        password: "Hotel@202",
        address: "Sai Baba Temple Road, Shirdi, MH 423109",
        gstNumber: "27EEFCS3456D4Z8",
        panNumber: "EEFCS3456D",
        registrationNumber: "MHHOTEL/2017/44556",
        licenseNumber: "LIC/SHR/2017/005",
        since: "2017-08-12",
      },
      {
        hotelName: "Rameswaram Beach Resort",
        ownerName: "Lakshmi Iyer",
        stateName: "Tamil Nadu",
        cityName: "Rameswaram",
        email: "beach.resort@email.com",
        phone: "+91 9876543215",
        password: "Hotel@303",
        address: "East Car Street, Rameswaram, TN 623526",
        gstNumber: "33FFGCS7890E5Z9",
        panNumber: "FFGCS7890E",
        registrationNumber: "TNHOTEL/2022/77889",
        licenseNumber: "LIC/RMS/2022/006",
        since: "2022-04-05",
      },
      {
        hotelName: "Tirupati Grand",
        ownerName: "Srinivas Reddy",
        stateName: "Andhra Pradesh",
        cityName: "Tirupati",
        email: "tirupati.grand@email.com",
        phone: "+91 9876543216",
        password: "Hotel@404",
        address: "Tirumala Road, Tirupati, AP 517501",
        gstNumber: "37GGHCS1234F6Z0",
        panNumber: "GGHCS1234F",
        registrationNumber: "APHOTEL/2016/99001",
        licenseNumber: "LIC/TPT/2016/007",
        since: "2016-12-01",
      },
      {
        hotelName: "Guruvayur Temple Stay",
        ownerName: "Krishna Menon",
        stateName: "Kerala",
        cityName: "Guruvayur",
        email: "temple.stay@email.com",
        phone: "+91 9876543217",
        password: "Hotel@505",
        address: "East Nada, Guruvayur, KL 680101",
        gstNumber: "32HHICS5678G7Z1",
        panNumber: "HHICS5678G",
        registrationNumber: "KLHOTEL/2020/22334",
        licenseNumber: "LIC/GUR/2020/008",
        since: "2020-07-18",
      },
    ]

    samplePartners.forEach((partner) => {
      const state = states.find((s) => s.name === partner.stateName)
      if (state) {
        const cities = locationDataStore.getCitiesByState(state.id)
        const city = cities.find((c) => c.name === partner.cityName)
        if (city) {
          this.addPartner({
            hotelName: partner.hotelName,
            ownerName: partner.ownerName,
            stateId: state.id,
            cityId: city.id,
            email: partner.email,
            phone: partner.phone,
            password: partner.password,
            address: partner.address,
            documents: {
              gstNumber: partner.gstNumber,
              panNumber: partner.panNumber,
              registrationNumber: partner.registrationNumber,
              licenseNumber: partner.licenseNumber,
              otherDocuments: ["Fire Safety Certificate", "Health License"],
            },
            since: partner.since,
          })
        }
      }
    })
  }

  getAllPartners(): HotelPartner[] {
    return Array.from(this.partners.values()).sort((a, b) => a.hotelName.localeCompare(b.hotelName))
  }

  getActivePartners(): HotelPartner[] {
    return this.getAllPartners().filter((p) => p.status === "active")
  }

  getInactivePartners(): HotelPartner[] {
    return this.getAllPartners().filter((p) => p.status === "inactive")
  }

  getPartnerById(id: string): HotelPartner | undefined {
    return this.partners.get(id)
  }

  getPartnerByEmail(email: string): HotelPartner | undefined {
    return Array.from(this.partners.values()).find((p) => p.email.toLowerCase() === email.toLowerCase())
  }

  addPartner(data: {
    hotelName: string
    ownerName: string
    stateId: string
    cityId: string
    email: string
    phone: string
    password: string
    address: string
    documents: HotelDocument
    since: string
  }): HotelPartner | null {
    // Check for duplicate email
    if (this.getPartnerByEmail(data.email)) {
      return null
    }

    const state = locationDataStore.getStateById(data.stateId)
    const city = locationDataStore.getCityById(data.cityId)

    if (!state || !city) return null

    this.idCounter++
    const id = `hotel_partner_${String(this.idCounter).padStart(3, "0")}`
    const now = new Date().toISOString().split("T")[0]

    const partner: HotelPartner = {
      id,
      hotelName: data.hotelName.trim(),
      ownerName: data.ownerName.trim(),
      stateId: data.stateId,
      stateName: state.name,
      cityId: data.cityId,
      cityName: city.name,
      email: data.email.trim().toLowerCase(),
      phone: data.phone.trim(),
      password: data.password,
      address: data.address.trim(),
      documents: data.documents,
      since: data.since,
      status: "active",
      createdAt: now,
      updatedAt: now,
    }

    this.partners.set(id, partner)
    return partner
  }

  updatePartner(
    id: string,
    data: {
      hotelName: string
      ownerName: string
      stateId: string
      cityId: string
      email: string
      phone: string
      password?: string
      address: string
      documents: HotelDocument
      since: string
    },
  ): HotelPartner | null {
    const partner = this.partners.get(id)
    if (!partner) return null

    // Check for duplicate email (excluding current partner)
    const existing = this.getPartnerByEmail(data.email)
    if (existing && existing.id !== id) {
      return null
    }

    const state = locationDataStore.getStateById(data.stateId)
    const city = locationDataStore.getCityById(data.cityId)

    if (!state || !city) return null

    const updatedPartner: HotelPartner = {
      ...partner,
      hotelName: data.hotelName.trim(),
      ownerName: data.ownerName.trim(),
      stateId: data.stateId,
      stateName: state.name,
      cityId: data.cityId,
      cityName: city.name,
      email: data.email.trim().toLowerCase(),
      phone: data.phone.trim(),
      password: data.password || partner.password,
      address: data.address.trim(),
      documents: data.documents,
      since: data.since,
      updatedAt: new Date().toISOString().split("T")[0],
    }

    this.partners.set(id, updatedPartner)
    return updatedPartner
  }

  toggleStatus(id: string): HotelPartner | null {
    const partner = this.partners.get(id)
    if (!partner) return null

    const updatedPartner: HotelPartner = {
      ...partner,
      status: partner.status === "active" ? "inactive" : "active",
      updatedAt: new Date().toISOString().split("T")[0],
    }

    this.partners.set(id, updatedPartner)
    return updatedPartner
  }

  deletePartner(id: string): boolean {
    return this.partners.delete(id)
  }
}

export const hotelPartnerDataStore = new HotelPartnerDataStore()
