export type LegalPageSlug =
  | "user-terms"
  | "privacy-policy"
  | "cancellation-policy"
  | "user-refund-policy"
  | "user-faq"
  | "agent-terms"
  | "hotel-terms"
  | "withdraw-terms"

export interface LegalPage {
  id: string
  slug: LegalPageSlug
  title: string
  metaTitle: string
  metaDescription: string
  content: string
  updatedAt: Date
  updatedBy: string
}

// In-memory data store for legal pages
const legalPagesStore: Map<LegalPageSlug, LegalPage> = new Map([
  [
    "user-terms",
    {
      id: "1",
      slug: "user-terms",
      title: "User Terms & Conditions",
      metaTitle: "User Terms & Conditions | Teertham",
      metaDescription:
        "Read the terms and conditions for using Teertham services. Understand your rights and responsibilities as a user of our platform.",
      content:
        "<h2>Terms of Service</h2><p>Welcome to Teertham. By using our services, you agree to these terms.</p><h3>1. Acceptance of Terms</h3><p>By accessing or using our platform, you agree to be bound by these Terms of Service.</p>",
      updatedAt: new Date("2024-01-15"),
      updatedBy: "Admin",
    },
  ],
  [
    "privacy-policy",
    {
      id: "2",
      slug: "privacy-policy",
      title: "Privacy Policy",
      metaTitle: "Privacy Policy | Teertham",
      metaDescription:
        "Learn how Teertham collects, uses, and protects your personal information. Your privacy is important to us.",
      content:
        "<h2>Privacy Policy</h2><p>At Teertham, we take your privacy seriously.</p><h3>Information We Collect</h3><p>We collect information you provide directly to us when you create an account or make a booking.</p>",
      updatedAt: new Date("2024-01-10"),
      updatedBy: "Admin",
    },
  ],
  [
    "cancellation-policy",
    {
      id: "3",
      slug: "cancellation-policy",
      title: "Cancellation Policy",
      metaTitle: "Cancellation Policy | Teertham",
      metaDescription:
        "Understand our cancellation policy for bookings. Learn about cancellation fees and refund timelines.",
      content:
        "<h2>Cancellation Policy</h2><p>We understand plans can change. Here is our cancellation policy.</p><h3>Hotel Cancellations</h3><p>Free cancellation up to 24 hours before check-in.</p>",
      updatedAt: new Date("2024-01-12"),
      updatedBy: "Admin",
    },
  ],
  [
    "user-refund-policy",
    {
      id: "4",
      slug: "user-refund-policy",
      title: "Refund Policy",
      metaTitle: "Refund Policy | Teertham",
      metaDescription:
        "Learn about our refund policy for cancelled bookings. Understand refund timelines and eligibility criteria.",
      content:
        "<h2>Refund Policy</h2><p>Your satisfaction is our priority.</p><h3>Refund Eligibility</h3><p>Refunds are processed within 5-7 business days of cancellation approval.</p>",
      updatedAt: new Date("2024-01-08"),
      updatedBy: "Admin",
    },
  ],
  [
    "user-faq",
    {
      id: "5",
      slug: "user-faq",
      title: "Frequently Asked Questions",
      metaTitle: "FAQ | Teertham",
      metaDescription: "Find answers to common questions about Teertham services, bookings, payments, and more.",
      content:
        "<h2>Frequently Asked Questions</h2><h3>How do I make a booking?</h3><p>You can make a booking by selecting your destination, dates, and preferred accommodation.</p><h3>What payment methods do you accept?</h3><p>We accept all major credit cards, debit cards, and UPI payments.</p>",
      updatedAt: new Date("2024-01-20"),
      updatedBy: "Admin",
    },
  ],
  [
    "agent-terms",
    {
      id: "6",
      slug: "agent-terms",
      title: "Agent Terms & Conditions",
      metaTitle: "Agent Terms & Conditions | Teertham",
      metaDescription:
        "Terms and conditions for travel agents partnering with Teertham. Understand commission structure and responsibilities.",
      content:
        "<h2>Agent Terms & Conditions</h2><p>Welcome to the Teertham Agent Partner Program.</p><h3>Commission Structure</h3><p>Agents earn commission on successful bookings made through their referral links.</p>",
      updatedAt: new Date("2024-01-05"),
      updatedBy: "Admin",
    },
  ],
  [
    "hotel-terms",
    {
      id: "7",
      slug: "hotel-terms",
      title: "Hotel Partner Terms",
      metaTitle: "Hotel Partner Terms | Teertham",
      metaDescription: "Terms and conditions for hotel partners listing their properties on Teertham platform.",
      content:
        "<h2>Hotel Partner Terms</h2><p>Thank you for partnering with Teertham.</p><h3>Listing Requirements</h3><p>All listed properties must meet our quality standards and safety requirements.</p>",
      updatedAt: new Date("2024-01-03"),
      updatedBy: "Admin",
    },
  ],
  [
    "withdraw-terms",
    {
      id: "8",
      slug: "withdraw-terms",
      title: "Withdrawal Terms",
      metaTitle: "Withdrawal Terms | Teertham",
      metaDescription:
        "Understand the terms for withdrawing earnings from Teertham. Learn about minimum withdrawal amounts and processing times.",
      content:
        "<h2>Withdrawal Terms</h2><p>Partners can withdraw their earnings subject to the following terms.</p><h3>Minimum Withdrawal</h3><p>The minimum withdrawal amount is ₹1,000.</p>",
      updatedAt: new Date("2024-01-01"),
      updatedBy: "Admin",
    },
  ],
])

/**
 * Get a legal page by slug
 */
export function getLegalPage(slug: LegalPageSlug): LegalPage | undefined {
  return legalPagesStore.get(slug)
}

/**
 * Get all legal pages
 */
export function getAllLegalPages(): LegalPage[] {
  return Array.from(legalPagesStore.values())
}

/**
 * Update a legal page
 */
export function updateLegalPage(
  slug: LegalPageSlug,
  data: {
    metaTitle: string
    metaDescription: string
    content: string
  },
): LegalPage {
  const existing = legalPagesStore.get(slug)
  if (!existing) {
    throw new Error(`Legal page with slug "${slug}" not found`)
  }

  const updated: LegalPage = {
    ...existing,
    metaTitle: data.metaTitle,
    metaDescription: data.metaDescription,
    content: data.content,
    updatedAt: new Date(),
    updatedBy: "Admin",
  }

  legalPagesStore.set(slug, updated)
  return updated
}
