export type LocationStatus = "active" | "inactive"

export interface State {
  id: string
  name: string
  status: LocationStatus
  createdAt: string
  updatedAt: string
}

export interface City {
  id: string
  stateId: string
  stateName: string
  name: string
  status: LocationStatus
  createdAt: string
  updatedAt: string
}

// In-memory location data store
class LocationDataStore {
  private states: Map<string, State> = new Map()
  private cities: Map<string, City> = new Map()
  private stateIdCounter = 0
  private cityIdCounter = 0

  constructor() {
    this.initializeSampleData()
  }

  private initializeSampleData() {
    const sampleStates: Omit<State, "id">[] = [
      { name: "Uttar Pradesh", status: "active", createdAt: "2024-01-15", updatedAt: "2024-01-15" },
      { name: "Uttarakhand", status: "active", createdAt: "2024-01-16", updatedAt: "2024-01-16" },
      { name: "Rajasthan", status: "active", createdAt: "2024-01-17", updatedAt: "2024-01-17" },
      { name: "Gujarat", status: "active", createdAt: "2024-01-18", updatedAt: "2024-01-18" },
      { name: "Maharashtra", status: "active", createdAt: "2024-01-19", updatedAt: "2024-01-19" },
      { name: "Tamil Nadu", status: "active", createdAt: "2024-01-20", updatedAt: "2024-01-20" },
      { name: "Karnataka", status: "inactive", createdAt: "2024-01-21", updatedAt: "2024-02-10" },
      { name: "Kerala", status: "active", createdAt: "2024-01-22", updatedAt: "2024-01-22" },
      { name: "Andhra Pradesh", status: "active", createdAt: "2024-01-23", updatedAt: "2024-01-23" },
      { name: "West Bengal", status: "inactive", createdAt: "2024-01-24", updatedAt: "2024-03-05" },
      { name: "Madhya Pradesh", status: "active", createdAt: "2024-02-01", updatedAt: "2024-02-01" },
      { name: "Bihar", status: "active", createdAt: "2024-02-05", updatedAt: "2024-02-05" },
    ]

    sampleStates.forEach((state) => {
      this.addState(state.name, state.status)
    })

    // Add sample cities
    const sampleCities: { stateName: string; cities: string[] }[] = [
      { stateName: "Uttar Pradesh", cities: ["Varanasi", "Ayodhya", "Mathura", "Prayagraj", "Lucknow"] },
      { stateName: "Uttarakhand", cities: ["Haridwar", "Rishikesh", "Dehradun", "Kedarnath", "Badrinath"] },
      { stateName: "Rajasthan", cities: ["Jaipur", "Pushkar", "Ajmer", "Udaipur", "Jodhpur"] },
      { stateName: "Gujarat", cities: ["Dwarka", "Somnath", "Ahmedabad", "Vadodara"] },
      { stateName: "Maharashtra", cities: ["Shirdi", "Nashik", "Pune", "Mumbai", "Kolhapur"] },
      { stateName: "Tamil Nadu", cities: ["Rameswaram", "Madurai", "Kanchipuram", "Chennai", "Thanjavur"] },
      { stateName: "Karnataka", cities: ["Udupi", "Hampi", "Mysore", "Bangalore"] },
      { stateName: "Kerala", cities: ["Guruvayur", "Sabarimala", "Thiruvananthapuram", "Kochi"] },
      { stateName: "Andhra Pradesh", cities: ["Tirupati", "Vijayawada", "Visakhapatnam", "Srisailam"] },
      { stateName: "West Bengal", cities: ["Kolkata", "Dakshineswar", "Tarapith", "Kalighat"] },
    ]

    sampleCities.forEach(({ stateName, cities }) => {
      const state = this.getStateByName(stateName)
      if (state) {
        cities.forEach((cityName) => {
          this.addCity(state.id, cityName)
        })
      }
    })
  }

  // State methods
  getAllStates(): State[] {
    return Array.from(this.states.values()).sort((a, b) => a.name.localeCompare(b.name))
  }

  getActiveStates(): State[] {
    return this.getAllStates().filter((s) => s.status === "active")
  }

  getStateById(id: string): State | undefined {
    return this.states.get(id)
  }

  getStateByName(name: string): State | undefined {
    return Array.from(this.states.values()).find((s) => s.name.toLowerCase() === name.toLowerCase())
  }

  addState(name: string, status: LocationStatus = "active"): State | null {
    // Check for duplicate
    if (this.getStateByName(name)) {
      return null
    }

    this.stateIdCounter++
    const id = `state_${String(this.stateIdCounter).padStart(3, "0")}`
    const now = new Date().toISOString().split("T")[0]

    const state: State = {
      id,
      name: name.trim(),
      status,
      createdAt: now,
      updatedAt: now,
    }

    this.states.set(id, state)
    return state
  }

  updateState(id: string, name: string): State | null {
    const state = this.states.get(id)
    if (!state) return null

    // Check for duplicate name (excluding current state)
    const existing = this.getStateByName(name)
    if (existing && existing.id !== id) {
      return null
    }

    const updatedState: State = {
      ...state,
      name: name.trim(),
      updatedAt: new Date().toISOString().split("T")[0],
    }

    this.states.set(id, updatedState)

    // Update state name in all associated cities
    this.cities.forEach((city, cityId) => {
      if (city.stateId === id) {
        this.cities.set(cityId, { ...city, stateName: updatedState.name })
      }
    })

    return updatedState
  }

  toggleStateStatus(id: string): State | null {
    const state = this.states.get(id)
    if (!state) return null

    const updatedState: State = {
      ...state,
      status: state.status === "active" ? "inactive" : "active",
      updatedAt: new Date().toISOString().split("T")[0],
    }

    this.states.set(id, updatedState)
    return updatedState
  }

  deleteState(id: string): boolean {
    const state = this.states.get(id)
    if (!state) return false

    // Delete all cities in this state first
    this.cities.forEach((city, cityId) => {
      if (city.stateId === id) {
        this.cities.delete(cityId)
      }
    })

    return this.states.delete(id)
  }

  // City methods
  getAllCities(): City[] {
    return Array.from(this.cities.values()).sort((a, b) => {
      const stateCompare = a.stateName.localeCompare(b.stateName)
      return stateCompare !== 0 ? stateCompare : a.name.localeCompare(b.name)
    })
  }

  getActiveCities(): City[] {
    return this.getAllCities().filter((c) => c.status === "active")
  }

  getCitiesByState(stateId: string): City[] {
    return this.getAllCities().filter((c) => c.stateId === stateId)
  }

  getCityById(id: string): City | undefined {
    return this.cities.get(id)
  }

  getCityByNameAndState(name: string, stateId: string): City | undefined {
    return Array.from(this.cities.values()).find(
      (c) => c.name.toLowerCase() === name.toLowerCase() && c.stateId === stateId,
    )
  }

  addCity(stateId: string, name: string, status: LocationStatus = "active"): City | null {
    const state = this.states.get(stateId)
    if (!state) return null

    // Check for duplicate in the same state
    if (this.getCityByNameAndState(name, stateId)) {
      return null
    }

    this.cityIdCounter++
    const id = `city_${String(this.cityIdCounter).padStart(3, "0")}`
    const now = new Date().toISOString().split("T")[0]

    const city: City = {
      id,
      stateId,
      stateName: state.name,
      name: name.trim(),
      status,
      createdAt: now,
      updatedAt: now,
    }

    this.cities.set(id, city)
    return city
  }

  updateCity(id: string, stateId: string, name: string): City | null {
    const city = this.cities.get(id)
    if (!city) return null

    const state = this.states.get(stateId)
    if (!state) return null

    // Check for duplicate name in the same state (excluding current city)
    const existing = this.getCityByNameAndState(name, stateId)
    if (existing && existing.id !== id) {
      return null
    }

    const updatedCity: City = {
      ...city,
      stateId,
      stateName: state.name,
      name: name.trim(),
      updatedAt: new Date().toISOString().split("T")[0],
    }

    this.cities.set(id, updatedCity)
    return updatedCity
  }

  toggleCityStatus(id: string): City | null {
    const city = this.cities.get(id)
    if (!city) return null

    const updatedCity: City = {
      ...city,
      status: city.status === "active" ? "inactive" : "active",
      updatedAt: new Date().toISOString().split("T")[0],
    }

    this.cities.set(id, updatedCity)
    return updatedCity
  }

  deleteCity(id: string): boolean {
    return this.cities.delete(id)
  }
}

export const locationDataStore = new LocationDataStore()
