export type VehicleStatus = "active" | "inactive"

export interface VehicleCategory {
  id: string
  name: string
  status: VehicleStatus
  createdAt: string
  updatedAt: string
}

export interface Vehicle {
  id: string
  name: string
  categoryId: string
  categoryName: string
  photo: string
  mileage: number // in km/l
  perKmPrice: number
  perDayPrice: number
  status: VehicleStatus
  createdAt: string
  updatedAt: string
}

// In-memory vehicle data store
class VehicleDataStore {
  private categories: Map<string, VehicleCategory> = new Map()
  private vehicles: Map<string, Vehicle> = new Map()
  private categoryIdCounter = 0
  private vehicleIdCounter = 0

  constructor() {
    this.initializeSampleData()
  }

  private initializeSampleData() {
    // Sample vehicle categories
    const sampleCategories: Omit<VehicleCategory, "id">[] = [
      { name: "Sedan", status: "active", createdAt: "2024-01-15", updatedAt: "2024-01-15" },
      { name: "SUV", status: "active", createdAt: "2024-01-16", updatedAt: "2024-01-16" },
      { name: "Hatchback", status: "active", createdAt: "2024-01-17", updatedAt: "2024-01-17" },
      { name: "MUV", status: "active", createdAt: "2024-01-18", updatedAt: "2024-01-18" },
      { name: "Tempo Traveller", status: "active", createdAt: "2024-01-19", updatedAt: "2024-01-19" },
      { name: "Bus", status: "active", createdAt: "2024-01-20", updatedAt: "2024-01-20" },
      { name: "Luxury", status: "inactive", createdAt: "2024-01-21", updatedAt: "2024-02-10" },
    ]

    sampleCategories.forEach((category) => {
      this.addCategory(category.name, category.status)
    })

    // Sample vehicles
    const sampleVehicles: {
      categoryName: string
      vehicles: Omit<Vehicle, "id" | "categoryId" | "categoryName" | "createdAt" | "updatedAt">[]
    }[] = [
      {
        categoryName: "Sedan",
        vehicles: [
          {
            name: "Honda City",
            photo: "/placeholder.svg?height=80&width=120",
            mileage: 18,
            perKmPrice: 12,
            perDayPrice: 2500,
            status: "active",
          },
          {
            name: "Maruti Dzire",
            photo: "/placeholder.svg?height=80&width=120",
            mileage: 22,
            perKmPrice: 10,
            perDayPrice: 2000,
            status: "active",
          },
          {
            name: "Hyundai Verna",
            photo: "/placeholder.svg?height=80&width=120",
            mileage: 17,
            perKmPrice: 13,
            perDayPrice: 2800,
            status: "active",
          },
        ],
      },
      {
        categoryName: "SUV",
        vehicles: [
          {
            name: "Toyota Fortuner",
            photo: "/placeholder.svg?height=80&width=120",
            mileage: 12,
            perKmPrice: 20,
            perDayPrice: 5500,
            status: "active",
          },
          {
            name: "Mahindra Scorpio",
            photo: "/placeholder.svg?height=80&width=120",
            mileage: 14,
            perKmPrice: 15,
            perDayPrice: 3500,
            status: "active",
          },
          {
            name: "Tata Safari",
            photo: "/placeholder.svg?height=80&width=120",
            mileage: 13,
            perKmPrice: 16,
            perDayPrice: 4000,
            status: "inactive",
          },
        ],
      },
      {
        categoryName: "Hatchback",
        vehicles: [
          {
            name: "Maruti Swift",
            photo: "/placeholder.svg?height=80&width=120",
            mileage: 24,
            perKmPrice: 8,
            perDayPrice: 1500,
            status: "active",
          },
          {
            name: "Hyundai i20",
            photo: "/placeholder.svg?height=80&width=120",
            mileage: 20,
            perKmPrice: 9,
            perDayPrice: 1800,
            status: "active",
          },
        ],
      },
      {
        categoryName: "MUV",
        vehicles: [
          {
            name: "Toyota Innova",
            photo: "/placeholder.svg?height=80&width=120",
            mileage: 15,
            perKmPrice: 16,
            perDayPrice: 4000,
            status: "active",
          },
          {
            name: "Maruti Ertiga",
            photo: "/placeholder.svg?height=80&width=120",
            mileage: 18,
            perKmPrice: 12,
            perDayPrice: 2800,
            status: "active",
          },
        ],
      },
      {
        categoryName: "Tempo Traveller",
        vehicles: [
          {
            name: "Force Tempo Traveller 12 Seater",
            photo: "/placeholder.svg?height=80&width=120",
            mileage: 10,
            perKmPrice: 22,
            perDayPrice: 6000,
            status: "active",
          },
          {
            name: "Force Tempo Traveller 17 Seater",
            photo: "/placeholder.svg?height=80&width=120",
            mileage: 9,
            perKmPrice: 25,
            perDayPrice: 7500,
            status: "active",
          },
        ],
      },
      {
        categoryName: "Bus",
        vehicles: [
          {
            name: "Mini Bus 20 Seater",
            photo: "/placeholder.svg?height=80&width=120",
            mileage: 8,
            perKmPrice: 30,
            perDayPrice: 10000,
            status: "active",
          },
          {
            name: "Luxury Bus 40 Seater",
            photo: "/placeholder.svg?height=80&width=120",
            mileage: 5,
            perKmPrice: 45,
            perDayPrice: 18000,
            status: "inactive",
          },
        ],
      },
    ]

    sampleVehicles.forEach(({ categoryName, vehicles }) => {
      const category = this.getCategoryByName(categoryName)
      if (category) {
        vehicles.forEach((vehicle) => {
          this.addVehicle({
            name: vehicle.name,
            categoryId: category.id,
            photo: vehicle.photo,
            mileage: vehicle.mileage,
            perKmPrice: vehicle.perKmPrice,
            perDayPrice: vehicle.perDayPrice,
            status: vehicle.status,
          })
        })
      }
    })
  }

  // Category methods
  getAllCategories(): VehicleCategory[] {
    return Array.from(this.categories.values()).sort((a, b) => a.name.localeCompare(b.name))
  }

  getActiveCategories(): VehicleCategory[] {
    return this.getAllCategories().filter((c) => c.status === "active")
  }

  getCategoryById(id: string): VehicleCategory | undefined {
    return this.categories.get(id)
  }

  getCategoryByName(name: string): VehicleCategory | undefined {
    return Array.from(this.categories.values()).find((c) => c.name.toLowerCase() === name.toLowerCase())
  }

  addCategory(name: string, status: VehicleStatus = "active"): VehicleCategory | null {
    if (this.getCategoryByName(name)) {
      return null
    }

    this.categoryIdCounter++
    const id = `vehicle_category_${String(this.categoryIdCounter).padStart(3, "0")}`
    const now = new Date().toISOString().split("T")[0]

    const category: VehicleCategory = {
      id,
      name: name.trim(),
      status,
      createdAt: now,
      updatedAt: now,
    }

    this.categories.set(id, category)
    return category
  }

  updateCategory(id: string, name: string): VehicleCategory | null {
    const category = this.categories.get(id)
    if (!category) return null

    const existing = this.getCategoryByName(name)
    if (existing && existing.id !== id) {
      return null
    }

    const updatedCategory: VehicleCategory = {
      ...category,
      name: name.trim(),
      updatedAt: new Date().toISOString().split("T")[0],
    }

    this.categories.set(id, updatedCategory)

    // Update category name in all associated vehicles
    this.vehicles.forEach((vehicle, vehicleId) => {
      if (vehicle.categoryId === id) {
        this.vehicles.set(vehicleId, { ...vehicle, categoryName: updatedCategory.name })
      }
    })

    return updatedCategory
  }

  toggleCategoryStatus(id: string): VehicleCategory | null {
    const category = this.categories.get(id)
    if (!category) return null

    const updatedCategory: VehicleCategory = {
      ...category,
      status: category.status === "active" ? "inactive" : "active",
      updatedAt: new Date().toISOString().split("T")[0],
    }

    this.categories.set(id, updatedCategory)
    return updatedCategory
  }

  deleteCategory(id: string): boolean {
    const category = this.categories.get(id)
    if (!category) return false

    // Delete all vehicles in this category
    this.vehicles.forEach((vehicle, vehicleId) => {
      if (vehicle.categoryId === id) {
        this.vehicles.delete(vehicleId)
      }
    })

    return this.categories.delete(id)
  }

  // Vehicle methods
  getAllVehicles(): Vehicle[] {
    return Array.from(this.vehicles.values()).sort((a, b) => a.name.localeCompare(b.name))
  }

  getActiveVehicles(): Vehicle[] {
    return this.getAllVehicles().filter((v) => v.status === "active")
  }

  getVehiclesByCategory(categoryId: string): Vehicle[] {
    return this.getAllVehicles().filter((v) => v.categoryId === categoryId)
  }

  getVehicleById(id: string): Vehicle | undefined {
    return this.vehicles.get(id)
  }

  getVehicleByName(name: string): Vehicle | undefined {
    return Array.from(this.vehicles.values()).find((v) => v.name.toLowerCase() === name.toLowerCase())
  }

  addVehicle(data: {
    name: string
    categoryId: string
    photo: string
    mileage: number
    perKmPrice: number
    perDayPrice: number
    status?: VehicleStatus
  }): Vehicle | null {
    const category = this.categories.get(data.categoryId)
    if (!category) return null

    // Check for duplicate vehicle name
    if (this.getVehicleByName(data.name)) {
      return null
    }

    this.vehicleIdCounter++
    const id = `vehicle_${String(this.vehicleIdCounter).padStart(3, "0")}`
    const now = new Date().toISOString().split("T")[0]

    const vehicle: Vehicle = {
      id,
      name: data.name.trim(),
      categoryId: data.categoryId,
      categoryName: category.name,
      photo: data.photo,
      mileage: data.mileage,
      perKmPrice: data.perKmPrice,
      perDayPrice: data.perDayPrice,
      status: data.status || "active",
      createdAt: now,
      updatedAt: now,
    }

    this.vehicles.set(id, vehicle)
    return vehicle
  }

  updateVehicle(
    id: string,
    data: {
      name: string
      categoryId: string
      photo: string
      mileage: number
      perKmPrice: number
      perDayPrice: number
    },
  ): Vehicle | null {
    const vehicle = this.vehicles.get(id)
    if (!vehicle) return null

    const category = this.categories.get(data.categoryId)
    if (!category) return null

    // Check for duplicate name (excluding current vehicle)
    const existing = this.getVehicleByName(data.name)
    if (existing && existing.id !== id) {
      return null
    }

    const updatedVehicle: Vehicle = {
      ...vehicle,
      name: data.name.trim(),
      categoryId: data.categoryId,
      categoryName: category.name,
      photo: data.photo,
      mileage: data.mileage,
      perKmPrice: data.perKmPrice,
      perDayPrice: data.perDayPrice,
      updatedAt: new Date().toISOString().split("T")[0],
    }

    this.vehicles.set(id, updatedVehicle)
    return updatedVehicle
  }

  toggleVehicleStatus(id: string): Vehicle | null {
    const vehicle = this.vehicles.get(id)
    if (!vehicle) return null

    const updatedVehicle: Vehicle = {
      ...vehicle,
      status: vehicle.status === "active" ? "inactive" : "active",
      updatedAt: new Date().toISOString().split("T")[0],
    }

    this.vehicles.set(id, updatedVehicle)
    return updatedVehicle
  }

  deleteVehicle(id: string): boolean {
    return this.vehicles.delete(id)
  }
}

export const vehicleDataStore = new VehicleDataStore()
