import DOMPurify from "isomorphic-dompurify"

/**
 * Sanitize HTML content to prevent XSS attacks
 */
export function sanitizeHtml(dirty: string): string {
  return DOMPurify.sanitize(dirty, {
    ALLOWED_TAGS: ["b", "i", "em", "strong", "a", "p", "br"],
    ALLOWED_ATTR: ["href", "target"],
  })
}

/**
 * Sanitize user input for safe storage
 * Protects against XSS, SQL injection patterns, and template injection
 */
export function sanitizeInput(input: string): string {
  if (typeof input !== "string") {
    return ""
  }

  return (
    input
      .trim()
      // Remove HTML tags
      .replace(/<[^>]*>/g, "")
      // Remove script injection patterns
      .replace(/javascript:/gi, "")
      .replace(/on\w+\s*=/gi, "")
      // Remove potential SQL injection patterns
      .replace(/['";\\]/g, "")
      .replace(/--/g, "")
      .replace(/\/\*/g, "")
      .replace(/\*\//g, "")
      // Remove template injection patterns
      .replace(/\{\{/g, "")
      .replace(/\}\}/g, "")
      .replace(/\$\{/g, "")
      // Normalize whitespace
      .replace(/\s+/g, " ")
  )
}

/**
 * Sanitize email input
 */
export function sanitizeEmail(email: string): string {
  if (typeof email !== "string") {
    return ""
  }
  return email
    .trim()
    .toLowerCase()
    .replace(/[^a-z0-9@._+-]/gi, "")
}

/**
 * Sanitize phone number input
 */
export function sanitizePhone(phone: string): string {
  if (typeof phone !== "string") {
    return ""
  }
  return phone.trim().replace(/[^0-9+\-\s()]/g, "")
}
