import type { Metadata } from "next"

interface PageMetadataOptions {
  title: string
  description: string
  path?: string
  type?: "website" | "article" | "product" | "profile"
  image?: string
  noIndex?: boolean
}

const SITE_NAME = "Teertham"
const SITE_URL = process.env.NEXT_PUBLIC_SITE_URL || "https://teertham.com"

/**
 * Generate comprehensive SEO metadata for pages
 */
export function generatePageMetadata({
  title,
  description,
  path = "",
  type = "website",
  image = "/og-image.jpg",
  noIndex = false,
}: PageMetadataOptions): Metadata {
  const fullTitle = `${title} | ${SITE_NAME}`
  const url = `${SITE_URL}${path}`
  const imageUrl = image.startsWith("http") ? image : `${SITE_URL}${image}`

  return {
    title: fullTitle,
    description,
    applicationName: SITE_NAME,
    ...(noIndex && {
      robots: {
        index: false,
        follow: false,
      },
    }),
    openGraph: {
      type,
      url,
      title: fullTitle,
      description,
      siteName: SITE_NAME,
      images: [
        {
          url: imageUrl,
          width: 1200,
          height: 630,
          alt: title,
        },
      ],
    },
    twitter: {
      card: "summary_large_image",
      title: fullTitle,
      description,
      images: [imageUrl],
    },
    alternates: {
      canonical: url,
    },
  }
}

/**
 * Generate JSON-LD structured data
 */
export function generateJsonLd(data: Record<string, unknown>) {
  return {
    __html: JSON.stringify({
      "@context": "https://schema.org",
      ...data,
    }),
  }
}
