import { CURRENCY_CODE, CURRENCY_LOCALE } from "@/lib/constants/admin"

/**
 * Format currency with Indian Rupee symbol and proper comma separation
 */
export function formatCurrency(amount: number): string {
  return new Intl.NumberFormat(CURRENCY_LOCALE, {
    style: "currency",
    currency: CURRENCY_CODE,
    maximumFractionDigits: 0,
  }).format(amount)
}

/**
 * Format currency without symbol (for calculations display)
 */
export function formatCurrencyNumber(amount: number): string {
  return new Intl.NumberFormat(CURRENCY_LOCALE, {
    maximumFractionDigits: 0,
  }).format(amount)
}

/**
 * Format date to display format (e.g., "Jan 15, 2024")
 */
export function formatDate(dateString: string): string {
  const date = new Date(dateString)
  return new Intl.DateTimeFormat(CURRENCY_LOCALE, {
    year: "numeric",
    month: "short",
    day: "numeric",
  }).format(date)
}

/**
 * Format date to long format (e.g., "January 15, 2024")
 */
export function formatDateLong(dateString: string): string {
  const date = new Date(dateString)
  return new Intl.DateTimeFormat(CURRENCY_LOCALE, {
    year: "numeric",
    month: "long",
    day: "numeric",
  }).format(date)
}

/**
 * Format date and time (e.g., "Jan 15, 2024 10:30 AM")
 */
export function formatDateTime(dateString: string): string {
  const date = new Date(dateString)
  return new Intl.DateTimeFormat(CURRENCY_LOCALE, {
    year: "numeric",
    month: "short",
    day: "numeric",
    hour: "numeric",
    minute: "numeric",
    hour12: true,
  }).format(date)
}

/**
 * Calculate discount percentage
 */
export function calculateDiscountPercentage(original: number, discounted: number): number {
  if (original <= 0) return 0
  return Math.round(((original - discounted) / original) * 100)
}

/**
 * Format days with proper pluralization
 */
export function formatDays(days: number): string {
  return days === 1 ? "1 day" : `${days} days`
}

/**
 * Format nights with proper pluralization
 */
export function formatNights(nights: number): string {
  return nights === 1 ? "1 night" : `${nights} nights`
}

/**
 * Format travelers count with proper pluralization
 */
export function formatTravelers(count: number): string {
  return count === 1 ? "1 traveler" : `${count} travelers`
}

/**
 * Format percentage
 */
export function formatPercentage(value: number, decimals = 0): string {
  return `${value.toFixed(decimals)}%`
}

/**
 * Truncate text with ellipsis
 */
export function truncateText(text: string, maxLength: number): string {
  if (text.length <= maxLength) return text
  return text.slice(0, maxLength) + "..."
}

/**
 * Format phone number (Indian format)
 */
export function formatPhoneNumber(phone: string): string {
  // Remove non-digit characters
  const cleaned = phone.replace(/\D/g, "")

  // Format as +91-XXXXX-XXXXX or similar
  if (cleaned.length === 10) {
    return `+91-${cleaned.slice(0, 5)}-${cleaned.slice(5)}`
  }

  return phone
}
