import { z } from "zod"
import DOMPurify from "isomorphic-dompurify"

/**
 * Sanitize HTML content to prevent XSS
 */
function sanitizeHtml(html: string): string {
  return DOMPurify.sanitize(html, {
    ALLOWED_TAGS: [
      "h1",
      "h2",
      "h3",
      "h4",
      "h5",
      "h6",
      "p",
      "br",
      "hr",
      "ul",
      "ol",
      "li",
      "strong",
      "em",
      "u",
      "s",
      "mark",
      "a",
      "blockquote",
      "code",
      "pre",
      "table",
      "thead",
      "tbody",
      "tr",
      "th",
      "td",
      "img",
      "figure",
      "figcaption",
      "div",
      "span",
    ],
    ALLOWED_ATTR: ["href", "target", "rel", "src", "alt", "class", "id", "title"],
  })
}

export const legalPageSchema = z.object({
  metaTitle: z
    .string()
    .min(10, "Meta title must be at least 10 characters")
    .max(70, "Meta title should be under 70 characters for SEO")
    .transform((val) => DOMPurify.sanitize(val.trim())),
  metaDescription: z
    .string()
    .min(50, "Meta description must be at least 50 characters")
    .max(160, "Meta description should be under 160 characters for SEO")
    .transform((val) => DOMPurify.sanitize(val.trim())),
  content: z
    .string()
    .min(100, "Content must be at least 100 characters")
    .transform((val) => sanitizeHtml(val)),
})

export type LegalPageInput = z.infer<typeof legalPageSchema>
