import { z } from "zod"

export const smtpSettingsSchema = z.object({
  host: z.string().min(1, "SMTP host is required"),
  port: z.coerce.number().min(1).max(65535, "Invalid port number"),
  username: z.string().min(1, "Username is required"),
  password: z.string().min(1, "Password is required"),
  encryption: z.enum(["none", "ssl", "tls"]),
  fromEmail: z.string().email("Invalid email address"),
  fromName: z.string().min(1, "From name is required"),
})

export const cashfreeSettingsSchema = z.object({
  appId: z.string().min(1, "App ID is required"),
  secretKey: z.string().min(1, "Secret key is required"),
  environment: z.enum(["sandbox", "production"]),
  webhookSecret: z.string().optional(),
})

export const agentCommissionSchema = z.object({
  defaultCommissionRate: z.coerce.number().min(0).max(100, "Commission rate must be between 0-100%"),
  minimumBookingValue: z.coerce.number().min(0, "Minimum booking value must be positive"),
  commissionType: z.enum(["percentage", "fixed"]),
  payoutCycle: z.enum(["weekly", "biweekly", "monthly"]),
})

export const withdrawSettingsSchema = z.object({
  minimumWithdrawAmount: z.coerce.number().min(0, "Must be positive"),
  maximumWithdrawAmount: z.coerce.number().min(0, "Must be positive"),
  processingFee: z.coerce.number().min(0, "Must be positive"),
  processingFeeType: z.enum(["percentage", "fixed"]),
  autoApproveThreshold: z.coerce.number().min(0, "Must be positive"),
  withdrawalLockPeriod: z.coerce.number().min(0, "Must be positive"),
})

export type SMTPSettingsInput = z.infer<typeof smtpSettingsSchema>
export type CashfreeSettingsInput = z.infer<typeof cashfreeSettingsSchema>
export type AgentCommissionInput = z.infer<typeof agentCommissionSchema>
export type WithdrawSettingsInput = z.infer<typeof withdrawSettingsSchema>
